package com.flybits.context.services;

import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.util.Log;

import com.flybits.commons.library.exceptions.PermissionNotSetException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.context.ContextScope;
import com.flybits.context.ReservedContextPlugin;
import com.flybits.context.models.ContextData;
import com.flybits.context.plugins.ContextPlugin;
import com.flybits.context.plugins.FlybitsContextPlugin;
import com.flybits.context.utils.ContextUtilities;
import com.google.android.gms.gcm.GcmNetworkManager;
import com.google.android.gms.gcm.GcmTaskService;
import com.google.android.gms.gcm.PeriodicTask;
import com.google.android.gms.gcm.TaskParams;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * The {@code FlybitsContextPluginService} class is the base class for all
 * {@link ContextPlugin}s. It can be used to create your own Context
 * Plugin using the {@link FlybitsContextPlugin.Builder#Builder(Class)} method or use one of the
 * pre-built options through {@link FlybitsContextPlugin.Builder#Builder(ReservedContextPlugin)}.
 */
public abstract class FlybitsContextPluginService extends GcmTaskService {

    private static final String _TAG = "ContextPluginService";
    private ScheduledExecutorService scheduleTaskExecutor;
    private long refreshTime;
    public static final String PREF_CP_REFRESH_RATE_START = "ContextPlugin-";

    @Override
    public void onCreate() {
        super.onCreate();
        Log.d(_TAG, this.getClass().getCanonicalName() + " : " + this.getClass().getSimpleName());
        if (!isSupported()) {
            stopSelf();
        }
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        initialize(intent.getExtras());
        refreshTime = intent.getLongExtra("minimumRefreshTime", 60);

        if (refreshTime < 60){
            long timeToRefresh  = (refreshTime < 10) ? 10 : refreshTime ;
            refreshContextData(timeToRefresh);
            return START_REDELIVER_INTENT;
        }

        return super.onStartCommand(intent, flags, startId);
    }

    @Override
    public void onDestroy() {
        if (scheduleTaskExecutor != null && !scheduleTaskExecutor.isShutdown()){
            scheduleTaskExecutor.shutdownNow();
        }

        super.onDestroy();
    }

    @Override
    public int onRunTask(TaskParams taskParams) {
        Logger.setTag(_TAG).d( "onRunTask: " + this.getClass().getSimpleName());
        if (checkPermissions(getBaseContext())) {

            initialize(taskParams.getExtras());
            ContextData data = getData();
            if (data != null) {
                long timeInSec = (data.getTime() > 0) ? data.getTime() : (System.currentTimeMillis() / 1000);
                data.update(getBaseContext(), timeInSec, null);
            }

            wait(2, TimeUnit.SECONDS);
        }
        return GcmNetworkManager.RESULT_SUCCESS;
    }

    public Context getContext(){
        return getBaseContext();
    }

    @Override
    public void onInitializeTasks() {
        Logger.setTag(_TAG).d( "onInitializeTask: " + this.getClass().getSimpleName());

        SharedPreferences preferences   = ContextScope.getContextPreferences(getBaseContext());
        long refreshTime                = preferences.getLong(FlybitsContextPlugin.getPrefName(this.getClass()), 3600);

        //TODO: This is hacky - But it needs to be there due to an error in versions prior to 0.14.3
        if (refreshTime == 3600){
            refreshTime = preferences.getLong(this.getClass().getCanonicalName(), 3600);
        }

        Bundle bundle                   = new Bundle();
        bundle.putLong("minimumRefreshTime", refreshTime);

        GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(getBaseContext());
        PeriodicTask.Builder task = new PeriodicTask.Builder()
                .setTag(this.getClass().getSimpleName())
                .setExtras(bundle)
                .setPersisted(true)
                .setUpdateCurrent(true)
                .setService(this.getClass())
                .setPeriod(refreshTime);
        mGcmNetworkManager.schedule(task.build());
    }

    /**
     * Get the {@link ContextData} that is associated to this {@link ContextPlugin}, each
     * {@link ContextPlugin} should have a corresponding {@link ContextData} object as this is
     * delivered to the Flybits Context Server.
     *
     * @return The {@link ContextData} object associated to this {@link ContextPlugin}.
     */
    public abstract ContextData getData();

    /**
     * Indicates how often the {@link ContextPlugin}'s data should be refreshed.
     *
     * @return the refresh time in seconds.
     */
    public long getRefreshTime(){
        return refreshTime;
    }

    /**
     * Gets the list of required permissions that should be declared within the application's
     * {@code AndroidManifest.xml} file in order of this {@link ContextPlugin} to function correctly.
     *
     * @return The list of permissions that should be declared within the application's
     * {@code AndroidManifest.xml} file.
     */
    public abstract String[] getRequiredPermissions();

    /**
     * Initializes the {@code FlybitsContextPluginService} so that it can initialize all the
     * necessary variables.
     *
     * @param bundle The bundle which may contain additional data needed for the
     *               {@code FlybitsContextPluginService} to correctly initialize itself.
     */
    public abstract void initialize(Bundle bundle);

    /**
     * Indicates whether or not the device's features support this
     * {@link ContextPlugin}. For example, if your device does not have
     * BlueTooth support than the {@link ReservedContextPlugin#BEACON} will not
     * be able to function correctly in this case. Each {@link ContextPlugin} should implement its
     * own checks to make sure the device supports all the necessary features.
     *
     * @return true if the device supports the various features needed for this
     * {@link ContextPlugin} to function correctly, false otherwise.
     */
    public abstract boolean isSupported();

    public void wait(long time, TimeUnit unit){

        long mServiceStartTime  = System.currentTimeMillis();
        long waitInMilli        = unit.toMillis(time);

        while (System.currentTimeMillis() - mServiceStartTime <= waitInMilli) {
            Thread.yield();
        }
    }

    private boolean checkPermissions(Context context) throws PermissionNotSetException {

        String[] listOfPermissions = getRequiredPermissions();
        for (String permission : listOfPermissions){
            if (!ContextUtilities.isPermissionGranted(context, permission)) {

                Log.e("Flybits", "You are attempting to register to this context plugin without enabling the " +
                        permission+" permission. This is not possible. Please include this permission in your manifest. " +
                        "For more information please visit https://developer.flybits.com/android-getting-started.html#context");
                return false;
            }
        }

        return true;
    }

    private void refreshContextData(long timeToRefresh){
        scheduleTaskExecutor = Executors.newScheduledThreadPool(1);
        scheduleTaskExecutor.scheduleAtFixedRate(new Runnable() {
            public void run() {
                new SeparateThread().start();
            }
        }, 0, timeToRefresh, TimeUnit.SECONDS);
    }

    private class SeparateThread extends Thread {

        SeparateThread() {}

        public void run() {
            ContextData data = getData();
            if (data != null) {
                long timeInSec                  = (data.getTime() > 0) ? data.getTime() : (System.currentTimeMillis() / 1000);
                data.update(getBaseContext(), timeInSec, null);
            }
        }
    }
}
