package com.flybits.context.plugins.network;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;
import android.telephony.TelephonyManager;

import com.flybits.context.models.ContextData;

import org.json.JSONException;
import org.json.JSONObject;

import static android.R.attr.type;

/**
 * The {@code NetworkData} class is used to define the network state of the user based at any given
 * time. As part of your network state a user the following information is obtained;
 *
 * <ul>
 *     <li>Whether or not the user is connected to the Internet</li>
 *     <li>If connected to Internet is through 2G, 3G, 4G, or WiFi.</li>
 *     <li>If connected to the Internet through WiFi, which SSID is the device connected to.</li>
 * </ul>
 */
public class NetworkData extends ContextData implements Parcelable {

    private NetworkConnectionType connectionType;
    private String ssid;
    private boolean isConnected;

    /**
     * Default constructor needed for generics instantiation
     */
    public NetworkData(){}

    /**
     * Constructor used for un-flattening a {@code NetworkData} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code NetworkData} parcel.
     */
    public NetworkData(Parcel in){
        super();
        this.connectionType     = NetworkConnectionType.fromKey(in.readInt());
        this.ssid               = in.readString();
        this.isConnected        = in.readInt()==1;
        setTime(in.readLong());
    }

    /**
     * Constructor used to define the {@code NetworkData} object based on preset values.
     *
     * @param ssid The SSID name of the WiFi that the device is connected.
     * @param isConnected Indicates whether or not the device is connected to the Internet.
     */
    public NetworkData(String ssid, boolean isConnected){
        super();
        this.connectionType     = NetworkConnectionType.WIFI;
        this.ssid               = ssid;
        this.isConnected        = isConnected;
    }

    /**
     * Constructor used to define the {@code NetworkData} object based on preset values.
     *
     * @param type Indicates the type of connection to the Internet (2G, 3G, 4G, WiFi, etc.)
     * @param ssid The SSID name of the WiFi that the device is connected.
     * @param isConnected Indicates whether or not the device is connected to the Internet.
     */
    public NetworkData(int type, String ssid, boolean isConnected){
        super();
        this.connectionType     = NetworkConnectionType.fromKey(type);
        this.ssid               = ssid;
        this.isConnected        = isConnected;
    }

    /**
     * Constructor used to define the attributes of the {@code NetworkData} object based on
     * {@link NetworkInfo} obtained from the device.
     *
     * @param mContext The current context of the application.
     * @param activeNetwork The {@link NetworkInfo} which provides information about the device's
     *                      network connection.
     */
    public NetworkData(@NonNull Context mContext, @NonNull NetworkInfo activeNetwork){
        super();
        if (mContext != null && activeNetwork != null){
            connectionType      = NetworkConnectionType.fromKey(getType(activeNetwork).getKey());
            isConnected         = isConnected(activeNetwork);
            ssid                = getSSID(mContext, activeNetwork);
        }else{
            connectionType      = NetworkConnectionType.UNKNOWN;
            isConnected         = false;
            ssid                = "";
        }
    }

    /**
     * Get the {@link NetworkConnectionType} of Internet connection that the device is connected 
     * through.
     *
     * @return Indicates the type of the network connection that is currently taking place between
     * the device and the Internet.
     */
    public NetworkConnectionType getConnectionType() {
        return connectionType;
    }

    /**
     * Get the SSID name of the WiFi AccessPoint to which the device is connected to the Internet
     * with.
     *
     * @return The SSID name or null if the device is not connected to a WiFi AccessPoint.
     */
    public String getSsid() {
        return ssid;
    }

    /**
     * Indicates whether or not the device is connected to the Internet.
     *
     * @return true if the user is connected to the Internet, false otherwise.
     */
    public boolean isConnected() {
        return isConnected;
    }

    @Override
    public String toJson() {

        JSONObject object=new JSONObject();
        try {
            object.put("connectionType", connectionType.getKey());
            object.put("ssid", ssid);
            object.put("isConnected", isConnected);
        }catch (JSONException exception){}

        return object.toString();
    }

    @Override
    public void fromJson(String json) {
        try {
            JSONObject jsonObj  = new JSONObject(json);
            connectionType      = NetworkConnectionType.fromKey(jsonObj.getInt("connectionType"));
            ssid                = jsonObj.getString("ssid");
            isConnected         = jsonObj.getBoolean("isConnected");

        }catch (JSONException exception){}
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        NetworkData that = (NetworkData) o;

        if (connectionType != that.connectionType) return false;
        if (!ssid.equals(that.ssid)) return false;
        return isConnected == that.isConnected;
    }

    @Override
    public String getPluginID() {
        return "ctx.sdk.network";
    }

    @Override
    public String toString(){
        return toJson();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeInt(connectionType.getKey());
        dest.writeString(ssid);
        dest.writeInt(isConnected? 1 : 0);
        dest.writeLong(getTime());
    }

    /**
     * Parcelable.Creator that instantiates {@code FitnessData} objects
     */
    public static final Creator<NetworkData> CREATOR = new Creator<NetworkData>() {
        public NetworkData createFromParcel(Parcel in) {
            return new NetworkData(in);
        }

        public NetworkData[] newArray(int size) {
            return new NetworkData[size];
        }
    };

    boolean isConnected(NetworkInfo activeNetwork){
        return activeNetwork != null && activeNetwork.isConnectedOrConnecting();
    }

    NetworkConnectionType getType(NetworkInfo activeNetwork){

        NetworkConnectionType connectionType = NetworkConnectionType.NONE;
        if (activeNetwork != null && activeNetwork.isConnected()){

            if (activeNetwork.getType() == ConnectivityManager.TYPE_MOBILE){
                connectionType = connectionType(activeNetwork.getSubtype());
            }else if (activeNetwork.getType() == ConnectivityManager.TYPE_WIFI){
                connectionType  = NetworkConnectionType.WIFI;
            }else{
                connectionType  = NetworkConnectionType.UNKNOWN;
            }
        }
        return connectionType;
    }

    String getSSID(Context context, NetworkInfo activeNetwork){

        String ssid    = "";
        if (activeNetwork != null && activeNetwork.isConnected()){
            if (activeNetwork.getType() == ConnectivityManager.TYPE_WIFI){
                ssid            = setSSID(context);
            }
        }
        if (ssid.startsWith("\"") && ssid.endsWith("\"")){
            ssid = ssid.substring(1, ssid.length()-1);
        }
        return ssid;
    }

    String setSSID(Context context){
        WifiManager wifiManager = (WifiManager) context.getSystemService(Context.WIFI_SERVICE);
        WifiInfo info = wifiManager.getConnectionInfo ();
        return info.getSSID();
    }

    NetworkConnectionType connectionType(int subType){
        switch(subType){
            case TelephonyManager.NETWORK_TYPE_1xRTT:
            case TelephonyManager.NETWORK_TYPE_CDMA:
            case TelephonyManager.NETWORK_TYPE_EDGE:
            case TelephonyManager.NETWORK_TYPE_GPRS:
            case TelephonyManager.NETWORK_TYPE_IDEN:
                return NetworkConnectionType._2G;
            case TelephonyManager.NETWORK_TYPE_EVDO_0:
            case TelephonyManager.NETWORK_TYPE_EVDO_A:
            case TelephonyManager.NETWORK_TYPE_HSDPA:
            case TelephonyManager.NETWORK_TYPE_HSPA:
            case TelephonyManager.NETWORK_TYPE_HSUPA:
            case TelephonyManager.NETWORK_TYPE_UMTS:
            case TelephonyManager.NETWORK_TYPE_EHRPD:
            case TelephonyManager.NETWORK_TYPE_EVDO_B:
            case TelephonyManager.NETWORK_TYPE_HSPAP:
                return NetworkConnectionType._3G;
            case TelephonyManager.NETWORK_TYPE_LTE:
                return NetworkConnectionType._4G;
            default:
                return NetworkConnectionType.UNKNOWN;
        }
    }
}
