package com.flybits.context;

import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.context.db.ContextDatabase;
import com.flybits.context.models.BasicData;
import com.flybits.context.models.ContextData;
import com.flybits.context.models.ContextPriority;
import com.flybits.context.plugins.ContextPlugin;
import com.flybits.context.utils.ContextUtilities;
import com.google.android.gms.gcm.GcmNetworkManager;
import com.google.android.gms.gcm.OneoffTask;
import com.google.android.gms.gcm.PeriodicTask;
import com.google.android.gms.gcm.Task;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

public class ContextManager {

    public static final String API_CONTEXT_DATA                    = ContextScope.ROOT+"/ctxdata";
    private static final String _TAG = "ContextManager";

    /**
     * Get Context Plug-in data that is stored within the Flybits Context DB. It's important to note
     * that {@code pluginID} and {@code subclass} must match, otherwise an empty {@link ContextData}
     * object will be returned or null if there was an error.
     *
     * @param context The context of the application.
     * @param pluginID The unique identifier that represents a specific {@link ContextPlugin}.
     * @param subclass The class that should be instantiated and context data should be parsed into.
     * @param callback The {@code ObjectResultCallback} that indicates whether or not the data for
     *                 {@code pluginID}, which could null.
     * @param <T> The generic that represents the a {@link ContextData} class which can parse
     *           the Context values.
     * @return  The {@link ContextData} that the {@link ContextPlugin} data is parsed into.
     */
    public static <T extends ContextData> ObjectResult<T> getData(@NonNull Context context,
                                                              @NonNull String pluginID,
                                                              @NonNull Class<T> subclass,
                                                              ObjectResultCallback<T> callback)
    throws java.security.InvalidParameterException{

        if (context == null){
            throw new java.security.InvalidParameterException("The Context parameter must not be null");
        }
        if (pluginID == null){
            throw new java.security.InvalidParameterException("The pluginId parameter must not be null");
        }
        if (subclass == null){
            throw new java.security.InvalidParameterException("The Class parameter must not be null");
        }

        try{
            T dataTest  = (T) subclass.newInstance();
        }catch (Exception e){
            if (e instanceof InstantiationException){
                throw new java.security.InvalidParameterException("The .Class file that will be instantiated " +
                        "must contain a constructor with no parameter for this method to be successfully processed");
            }
        }
        return getDataPrivate(context, pluginID, subclass, callback);
    }

    /**
     * Retrieves all the Context information from the local DB and sends it to the server if it is
     * determined that new Context data has been received.
     *
     * @param context The state of the application.
     * @param callback The {@code BasicResultCallback} that indicates whether or not the request was
     *                 successful or not. If not, a reason will be provided.
     * @return The {@code BasicResult} which contains information about the request being made.
     */
    public static BasicResult flushContextData(@NonNull final Context context, final BasicResultCallback callback){
        final Handler handler                   = new Handler(Looper.getMainLooper());
        final ExecutorService executorService   = Executors.newSingleThreadExecutor();
        final BasicResult basicResult      = new BasicResult(callback, handler, executorService);

        executorService.execute(new Runnable() {
            public void run() {

                boolean isSuccessful    = flushContextData(context);
                Result tempStatus = null;
                if (isSuccessful){
                    tempStatus  = new Result(200, "");
                }else{
                    tempStatus  = new Result(new FlybitsException("Error Updating Context"), "");
                }
                final Result resultStatus = tempStatus;
                basicResult.setResult(resultStatus);
            }
        });
        return basicResult;
    }

    /**
     * TODO: Comments
     * @param context
     * @return
     */
    public static boolean flushContextData(@NonNull Context context){

        if (context == null){
            return false;
        }

        boolean isSuccessful = false;

        List<BasicData> dataPlugins = ContextDatabase.getDatabase(context).basicDataDao().getAllNotSent();
        String jsonToSend = ContextUtilities.getContextData(dataPlugins);

        if (jsonToSend != null && jsonToSend.length() > 2) {
            try {
                final Result result = FlyAway.post(context, API_CONTEXT_DATA, jsonToSend, null, "FlushUtilities.flushContext", null);
                if (result.getStatus() == RequestStatus.COMPLETED) {

                    for (BasicData data : dataPlugins){
                        data.setSent(true);
                        ContextDatabase.getDatabase(context).basicDataDao().update(data);
                    }

                    isSuccessful = true;
                } else {
                    Logger.setTag(_TAG).e("Updated FlybitsContext in Server: Failed");
                }
            } catch (FlybitsException e) {
                Logger.exception("FlushUtilities.flushContext", e);
            }
        }

        return isSuccessful;
    }

    /**
     * Refresh the data that is associated to the {@link ContextPlugin}, this may make a network
     * request, open a new activity/dialog, collect information from a sensor and/or anything else
     * that the {@link ContextPlugin} has be designed to do.
     *
     * @param mContext The Context of the Application.
     * @param plugin The {@link ContextPlugin} that should be refreshed.
     *
     * Although this method seems a bit redundant, as it simply call the onRefresh of the Context
     * Plugin it has been written this way to be consistent with the other Flybits platforms
     */
    public static void refresh(Context mContext, ContextPlugin plugin){
        plugin.onRefresh(mContext);
    }

    /**
     * Begin the {@link ContextPlugin} that is responsible for collecting Contextual Data about the
     * user or device.
     *
     * @param mContext The Context of the Application.
     * @param plugin The {@link ContextPlugin} that should be started.
     *
     * Although this method seems a bit redundant, as it simply call the onStart of the Context
     * Plugin it has been written this way to be consistent with the other Flybits platforms
     */
    public static void start(Context mContext, ContextPlugin plugin){

        plugin.onStart(mContext);
    }

    /**
     * Stop the {@link ContextPlugin} that is responsible for collecting Contextual Data about the
     * user or device.
     *
     * @param mContext The Context of the Application.
     * @param plugin The {@link ContextPlugin} that should be stopped.
     *
     * Although this method seems a bit redundant, as it simply call the onStop of the Context
     * Plugin it has been written this way to be consistent with the other Flybits platforms
     */
    public static void stop(Context mContext, ContextPlugin plugin){
        plugin.onStop(mContext);
    }

    public static void registerForRules(final Context mContext, long timeInSeconds, long timeInSecondsFlex, TimeUnit unit){

        GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);
        PeriodicTask.Builder task = new PeriodicTask.Builder()
                .setTag("Context_Rules")
                .setUpdateCurrent(true)
                .setPersisted(true)
                .setPeriod(unit.toSeconds(timeInSeconds))
                .setService(ContextRulesService.class);
        mGcmNetworkManager.schedule(task.build());
        Logger.setTag(_TAG).d("Activated: Context_Rules, Time to Refresh: " + timeInSeconds);
    }

    static void registerForPluginUpdates(final Context mContext, HashMap<String, String> listOfClass, long timeToRefreshInSec){

        int listOfClassSize                     = listOfClass.size();
        SharedPreferences.Editor preferences    = ContextScope.getContextPreferences(mContext).edit();
        preferences.putInt(ContextPluginsService.PREF_KEY_NUM_CPS, listOfClassSize);
        preferences.putLong(ContextPluginsService.PREF_KEY_REFRESH_CPS, timeToRefreshInSec);

        Bundle bundle = new Bundle();
        bundle.putInt("listOfItems", listOfClassSize);
        int i = 0;

        Iterator it = listOfClass.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<String, String> pair = (Map.Entry<String, String>) it.next();

            String key          = pair.getKey();
            bundle.putString("item"+i, key);
            preferences.putString("item"+i, key);

            String value        = pair.getValue();
            bundle.putString("class"+i, value);
            bundle.putString("value"+i, value);
            it.remove();
            i++;
        }

        preferences.apply();
        GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);
        OneoffTask oneTimeTask = new OneoffTask.Builder()
                .setService(ContextPluginsService.class)
                .setTag("Context_Plugin_Once")
                .setExecutionWindow(0L, 30L)
                .setRequiredNetwork(Task.NETWORK_STATE_ANY)
                .build();

        mGcmNetworkManager.schedule(oneTimeTask);

        PeriodicTask.Builder task = new PeriodicTask.Builder()
                .setExtras(bundle)
                .setTag("Context_Plugin")
                .setUpdateCurrent(true)
                .setPersisted(true)
                .setPeriod(timeToRefreshInSec)
                .setService(ContextPluginsService.class);
        mGcmNetworkManager.schedule(task.build());


        Logger.setTag(_TAG).d("Activated: Context_Plugin - Time to Refresh: 90");
    }

    static void registerUploadingContext(final Context mContext, ContextPriority priority, long timeInSeconds, long timeInSecondsFlex, TimeUnit unit){

        Bundle bundle = new Bundle();
        bundle.putInt("priority", priority.getKey());

        GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);
        PeriodicTask.Builder task = new PeriodicTask.Builder()
                .setExtras(bundle)
                .setTag("Context_Uploading")
                .setUpdateCurrent(true)
                .setPersisted(true)
                .setRequiredNetwork(PeriodicTask.NETWORK_STATE_CONNECTED)
                .setPeriod(unit.toSeconds(timeInSeconds))
                .setService(ContextUploadingService.class);
        mGcmNetworkManager.schedule(task.build());
        Logger.setTag(_TAG).d("Activated: Context_Uploading with Priority: " + priority.getKey() + ", Time to Refresh: " + timeInSeconds);
    }

    static void unregisterFromRuleCollection(final Context mContext){

        Intent intent = new Intent(mContext, ContextRulesService.class);
        if (ContextUtilities.isServiceDefined(mContext, intent)) {
            GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);

            try {
                mGcmNetworkManager.cancelTask("Context_Rules", ContextRulesService.class);
                Logger.setTag(_TAG).d("UnActivated: Context_Rules");
            } catch (IllegalArgumentException ex) {
                Logger.exception("ContextManager.unregisterFromRuleCollection",ex);
            }
        }
        Logger.setTag(_TAG).d("unregisterFromRuleCollection Completed");
    }

    static void unregisterUploadingContext(final Context mContext){

        Intent intent = new Intent(mContext, ContextUploadingService.class);
        if (ContextUtilities.isServiceDefined(mContext, intent)) {
            GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);
            try {
                mGcmNetworkManager.cancelTask("Context_Uploading", ContextUploadingService.class);
                Logger.setTag(_TAG).d("UnActivated: Context_Uploading");
            } catch (IllegalArgumentException ex) {
                Logger.exception("ContextManager.unregisterUploadingContext",ex);
            }
        }
        Logger.setTag(_TAG).d("unregisterUploadingContext Completed");
    }

    static void unregisterPluginContext(final Context mContext){

        Intent intent = new Intent(mContext, ContextPluginsService.class);
        if (ContextUtilities.isServiceDefined(mContext, intent)) {
            GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);
            try {
                mGcmNetworkManager.cancelTask("Context_Plugin", ContextPluginsService.class);
                Logger.setTag(_TAG).d("UnActivated: Context_Plugin");
            } catch (IllegalArgumentException ex) {
                Logger.exception("ContextManager.unregisterPluginContext",ex);
            }
        }
        Logger.setTag(_TAG).d("ContextManager.unregisterPluginContext Completed");
    }

    static @Nullable <T extends ContextData> ObjectResult<T> getDataPrivate(@NonNull final Context context,
                                                                            @NonNull final String pluginID,
                                                                            @NonNull final Class<T> subclass,
                                                                            ObjectResultCallback<T> callback) {

        final Handler handler                   = new Handler(Looper.getMainLooper());
        final ExecutorService executorService   = Executors.newSingleThreadExecutor();
        final ObjectResult<T> objectResult      = new ObjectResult<T>(callback, handler, executorService);

        executorService.execute(new Runnable() {
            public void run() {

                BasicData dataFromCursor    = ContextDatabase.getDatabase(context).basicDataDao().get(pluginID);
                if (dataFromCursor != null && dataFromCursor.getValueAsString() != null) {
                    try {
                        T data = (T) subclass.newInstance();
                        data.fromJson(dataFromCursor.getValueAsString());
                        data.setTime(dataFromCursor.getTimestamp());

                        final Result<T> result   = new Result<T>(200, "");
                        result.setResponse(data);
                        objectResult.setResult(result);
                    }catch(final Exception e){
                        objectResult.setResult(new Result(new FlybitsException(e.getMessage()),""));
                        Logger.exception("ContextManager.getData", e);
                    }
                }
            }
        });
        return objectResult;
    }
}
