package com.flybits.context.models;

import android.arch.persistence.room.ColumnInfo;
import android.arch.persistence.room.Entity;
import android.arch.persistence.room.Ignore;
import android.arch.persistence.room.PrimaryKey;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.context.deserializations.DeserializeEvaluatedRules;
import com.flybits.context.models.results.RulesEvaluatedResult;
import com.flybits.context.utils.RulesQueryParameters;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * This class is used to represent an evaluated Context {@code Rule} within Flybits. A rule can be
 * either evaluated as true or false depending on values obtained from the Context Plugins which can
 * be obtained from a local sensor or an external one. A {@code Rule} needs to contain at least one
 * condition which needs to be satisfied to be true. It is important to note that a
 * {@code RuleEvaluated} is a simplified version of the {@link com.flybits.context.models.Rule}
 * object with an evaluation.
 *
 * <p>For example, an application developer may create a Context {@code RuleEvalutated} "Tired" that
 * is comprised of 2 context plugins; Step Counter and Sleep Quality. Therefore, an application
 * developer can create the "Tired" rule when a user's step counter is &gt; 10000 and
 * sleep quality is &lt; 6hrs. </p>
 */
@Entity(tableName = "ruleEvaluations")
public class RuleEvalutated implements Parcelable {

    public static final String API = Rule.API+"/evaluations";

    @PrimaryKey
    @ColumnInfo(name = "ruleEvaluationID")
    @NonNull
    private String id;

    @ColumnInfo(name = "name")
    private String name;

    @ColumnInfo(name = "isLastResult")
    private boolean lastResult;

    @Ignore
    private String dataAsString;

    @Ignore
    private long evaluatedAt;

    public RuleEvalutated(){}

    /**
     * Constructor used to define a {@code Rule} object.
     *
     * @param id The unique identifier that is used to identify a specific {@code Rule}.
     * @param name The name of the {@code Rule}.
     * @param lastResult Indicates whether or not the {@code Rule} is evaluated as true
     */
    @Ignore
    public RuleEvalutated(String id, String name, boolean lastResult){
        this.id             = id;
        this.lastResult     = lastResult;
        this.name           = name;
    }

    /**
     * Constructor used to define a {@code Rule} object.
     *
     * @param id The unique identifier that is used to identify a specific {@code Rule}.
     * @param lastEvaluated Indicates when the {@code Rule} was last evaluated in Epoch time.
     * @param lastResult Indicates whether or not the {@code Rule} is evaluated as true or false.
     * @param name The name of the {@code Rule}.
     * @param dataAsString The composition of the {@code Rule} as a String representation.
     */
    @Ignore
    public RuleEvalutated(String id, long lastEvaluated, boolean lastResult, String name, String dataAsString){
        this(id, name, lastResult);
        this.evaluatedAt = lastEvaluated;
        this.dataAsString   = dataAsString;
    }

    /**
     * Constructor used for un-flattening a {@code RuleEvalutated} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code RuleEvalutated} parcel.
     */
    @Ignore
    public RuleEvalutated(Parcel in){
        id                  = in.readString();
        evaluatedAt = in.readLong();
        lastResult          = in.readInt() == 1;
        name                = in.readString();
        dataAsString        = in.readString();
    }

    /**
     * Get the data composition of the {@code Rule} and the variables used to create the
     * {@code Rule}.
     *
     * @return The String representation of the composition of the {@code Rule}.
     */
    public String getDataAsString() {
        return dataAsString;
    }

    /**
     * Get the time, as epoch, when the {@code Rule} was last evaluated.
     *
     * @return The epoch time when the {@code Rule} was last evaluated.
     */
    public long getEvaluatedTime() {
        return evaluatedAt;
    }

    /**
     * Get the unique identifier for the {@code Rule}.
     *
     * @return The String representation of the {@code Rule}.
     */
    public String getId() {
        return id;
    }

    /**
     * The name of the {@code Rule} given by the user who created it.
     *
     * @return The name of the {@code Rule}.
     */
    public String getName() {
        return name;
    }

    /**
     * Indicates whether or not the {@code Rule} is currently true based on the contextual values
     * associated to the {@code Rule}.
     *
     * @return true if the {@code Rule} is evaluated at true, false otherwise.
     */
    public boolean isLastResult() {
        return lastResult;
    }

    /**
     * Set the unique identifier of the {@code RuleEvaluated}.
     *
     * @param id The String representation of the unique identifier.
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * Indicate whether or not the {@link Rule} is evaluated as true or false.
     *
     * @param lastResult true if the {@link Rule} is evaluated as true, false otherwise.
     */
    public void setLastResult(boolean lastResult) {
        this.lastResult = lastResult;
    }

    /**
     * Set the name of the {@link Rule}.
     *
     * @param name The String representation of the {@link Rule}.
     */
    public void setName(String name) {
        this.name = name;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        RuleEvalutated rule = (RuleEvalutated) o;

        return !(id != null ? !id.equals(rule.id) : rule.id != null);
    }

    @Override
    public String toString() {
        return "[ " + id + ", " + name + " : " + lastResult + " ]";
    }

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    @Override
    public int describeContents() {
        return 0;
    }

    /**
     * Flatten this {@code GeoPoint} into a Parcel.
     *
     * @param out The Parcel in which the {@code GeoPoint} object should be written.
     * @param flags Additional flags about how the DateOfBirth object should be written.
     * May be 0 or {@link #PARCELABLE_WRITE_RETURN_VALUE}.
     */
    @Override
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(id);
        out.writeLong(evaluatedAt);
        out.writeInt(lastResult? 1 : 0);
        out.writeString(name);
        out.writeString(dataAsString);
    }

    /**
     * Parcelable.Creator that instantiates {@code Rule} objects
     */
    public static final Creator<RuleEvalutated> CREATOR = new Creator<RuleEvalutated>(){
        public RuleEvalutated createFromParcel(Parcel in){
            return new RuleEvalutated(in);
        }

        public RuleEvalutated[] newArray(int size){
            return new RuleEvalutated[size];
        }
    };

    /**
     * A request to get a list of {@code Rule}s based on the {@code QueryParameters} parameter which
     * defines which options should be used to filter the list of returned parameters.
     *
     * @param mContext The context of the application.
     * @param params The {@code QueryParameters} that indicate filters that should be placed on
     *               the request.
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     *
     * @return The {@code RulesResult} that contains the network request for getting a list of
     * {@code Rule}s.
     */
    public static RulesEvaluatedResult get(final Context mContext, final RulesQueryParameters params,
                                  final PagedResultCallback<RuleEvalutated> callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final RulesEvaluatedResult result = new RulesEvaluatedResult(mContext, params, callback, executorService, handler);
        executorService.execute(new Runnable() {
            public void run() {
                try{

                    final Result<PagedResponse<RuleEvalutated>> getRules = FlyAway.get(mContext, API, params, new DeserializeEvaluatedRules(), "RuleEvaluated.get");
                    result.setResult(getRules, params);
                }catch (final FlybitsException e){
                    result.setFailed(e);
                }
            }
        });
        return result;
    }
}