package com.flybits.context.models;

import android.support.annotation.Nullable;
import android.util.Base64;

import com.flybits.context.ReservedContextPlugin;
import com.google.android.gms.maps.model.LatLng;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.InvalidParameterException;
import java.util.HashMap;
import java.util.UUID;

/**
 * The {@code RulePredicate} class is responsible for create a base predicate which is the root
 * entity needed for creating a Flybits Rule. A predicate is a statement that can be evaluated to a
 * boolean value also known as atomic formula. it Syntax is as follows; [Name]([arg1],[arg2],…)
 * Some Example include:
 * <ul>
 *     <li>old(john)</li>
 *     <li>gr(3,4)</li>
 *     <li>cold()</li>
 *     <li>nearBy(X,John)</li>
 * </ul>
 */
public class RulePredicate {

    private static final double EARTH_RADIUS = 6371009;
    private String predicate;
    private HashMap<String, String> pluginData;

    private RulePredicate(String value) {
        predicate   = value;
        pluginData  = new HashMap<>();
    }

    /**
     * Get the String representation of the {@code RulePredicate}.
     *
     * @return The string representation of the {@code RulePredicate}.
     */
    public String getString(){
        return predicate;
    }

    /**
     * Get the HashMap containing the plugin data that should be added to the rule.
     *
     * @return The HashMap associated to the plugin data.
     */
    public HashMap<String, String> getPluginData(){
        return pluginData;
    }

    /**
     * Add a key and value pair to the {@code PluginData} component of the {@link Rule}.
     *
     * @param key The {@code key} that should be used to store a {@code value} under.
     * @param value The {@code value} associated to the {@code key}.
     */
    public void addPluginData(String key, String value){

        if (pluginData == null){
            pluginData  = new HashMap<>();
        }

        pluginData.put(key, value);
    }

    /*
        START Numeric Predicate
     */

    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value is greater than the
     * numeric {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The numeric value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate greaterThan(String plugin, int value){
        return new RulePredicate("gr(" + plugin + "," + value + ")");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value is greater than the
     * numeric {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The numeric value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate greaterThan(String plugin, int value, String id) throws IllegalArgumentException{

        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return greaterThan(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value is greater than or equal
     * to the numeric {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The numeric value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate greaterThanOrEqual(String plugin, int value){
        return new RulePredicate("greq(" + plugin + "," + value + ")");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value is greater than or equal
     * the numeric {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The numeric value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
     public static RulePredicate greaterThanOrEqual(String plugin, int value, String id) throws IllegalArgumentException{
        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return RulePredicate.greaterThanOrEqual(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value is less than the numeric
     * {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The numeric value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate lessThan(String plugin, int value){
        return new RulePredicate("ls(" + plugin + "," + value + ")");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value is less than the numeric
     * {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The numeric value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate lessThan(String plugin, int value, String id) throws IllegalArgumentException{
        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return RulePredicate.lessThan(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value is less than or equal to
     * numeric {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The numeric value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate lessThanOrEqual(String plugin, int value){
        return new RulePredicate("lseq(" + plugin + "," + value + ")");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value is less than or equal to
     * numeric {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The numeric value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate lessThanOrEqual(String plugin, int value, String id) throws IllegalArgumentException{
        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return RulePredicate.lessThanOrEqual(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value is equal to the numeric
     * {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The numeric value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate equals(String plugin, int value){
        return new RulePredicate("eq(" + plugin + "," + value + ")");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value is equal to the numeric
     * {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The numeric value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate equals(String plugin, int value, String id) throws IllegalArgumentException{
        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return RulePredicate.equals(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value is not equal to the
     * numeric {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The numeric value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate notEquals(String plugin, int value){
        return new RulePredicate("neq(" + plugin + "," + value + ")");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value is not equal to the
     * numeric {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The numeric value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate notEquals(String plugin, int value, String id) throws IllegalArgumentException{
        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return RulePredicate.notEquals(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }
    /*
        END Numeric Comparison
     */


    /*
        START Boolean Predicate
     */
    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value is equal to the boolean
     * {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The boolean value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate equals(String plugin, boolean value){
        return new RulePredicate("boolEq(" + plugin + "," + value + ")");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value is equal to the boolean
     * {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The boolean value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate equals(String plugin, boolean value, String id) throws IllegalArgumentException{
        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return RulePredicate.equals(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }


    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value is not equal to the
     * boolean {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The boolean value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate notEquals(String plugin, boolean value){
        return new RulePredicate("boolNeq(" + plugin + "," + value + ")");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value is not equal to the
     * boolean {@code value}.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The boolean value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate notEquals(String plugin, boolean value, String id) throws IllegalArgumentException{
        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return RulePredicate.notEquals(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }
    /*
        END Boolean Predicate
     */

    /*
        START String Predicate
     */
    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value is equal to the String
     * {@code value}. This value is case-sensitive.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The String value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate equals(String plugin, String value){
        return new RulePredicate("stringEq(" + plugin + ",'" + value + "')");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value is equal to the String
     * {@code value}. This value is case-sensitive.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The String value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate equals(String plugin, String value, String id) throws IllegalArgumentException{
        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return RulePredicate.equals(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value is equal to the String
     * {@code value}. This value is case-insensitive.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The String value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate equalsIgnoreCase(String plugin, String value){
        return new RulePredicate("stringFoldEq(" + plugin + ",'" + value + "')");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value is equal to the String
     * {@code value}. This value is case-insensitive.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The String value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate equalsIgnoreCase(String plugin, String value, String id) throws IllegalArgumentException{
        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return RulePredicate.equalsIgnoreCase(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value is not equal to the
     * String {@code value}. This value is case-sensitive.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}
     * @param value The String value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate notEquals(String plugin, String value){
        return new RulePredicate("stringNeq(" + plugin + ",'" + value + "')");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value is not equal to the
     * String {@code value}. This value is case-sensitive.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The String value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate notEquals(String plugin, String value, String id) throws IllegalArgumentException{
        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return RulePredicate.notEquals(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value is not equal to the
     * String {@code value}. This value is case-insensitive.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The String value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate notEqualsIgnoreCase(String plugin, String value){
        return new RulePredicate("stringFoldNeq(" + plugin + ",'" + value + "')");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value is not equal to the
     * String {@code value}. This value is case-insensitive.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The String value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate notEqualsIgnoreCase(String plugin, String value, String id) throws IllegalArgumentException{
        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return RulePredicate.notEqualsIgnoreCase(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value contains the String
     * {@code value}. This value is case-sensitive.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The String value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate contains(String plugin, String value){
        return new RulePredicate("stringContains(" + plugin + ",'" + value + "')");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value contains the String
     * {@code value}. This value is case-sensitive.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The String value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate contains(String plugin, String value, String id) throws IllegalArgumentException{
        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return RulePredicate.contains(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plug-in} value contains the String
     * {@code value}. This value is case-insensitive.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The String value that should be used for comparison.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate containsIgnoreCase(String plugin, String value){
        return new RulePredicate("stringFoldContains(" + plugin + ",'" + value + "')");
    }

    /**
     * Creates a predicate that is true if the {@code Context Plugin} value contains the String
     * {@code value}. This value is case-insensitive.
     *
     * @param plugin The String representation of the Context Plug-in that should be evaluated for
     *               example, {@link ReservedContextPlugin#BATTERY}'s {@code isCharging} attribute.
     * @param value The String value that should be used for comparison.
     * @param id The unique identifier representing a specific entity that the {@link Rule} should
     *           be processed for.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    public static RulePredicate containsIgnoreCase(String plugin, String value, String id) throws IllegalArgumentException{
        String predicate    = getQueryParam(plugin, id);
        if (predicate != null){
            return RulePredicate.containsIgnoreCase(predicate, value);
        }
        throw new IllegalArgumentException("You plugin must be in the following format: w.x.y.z where the combination of w.x.y.z indicate the plugin identifier that should be evaluated.");
    }
    /*
    Start Location Predicate
     */
    /**
     * Creates a predicate that is true if the device's location is within a certain distance
     * (defined using the {@code distanceInMeters} attribute) of a geographical point.
     *
     * @param pointLat The latitude of a geographical point that the device's latitude should be
     *                 compared to.
     * @param pointLng The longitude of a geographical point that the device's longitude should be
     *                 compared to.
     * @param distanceInMeters The distance in meters that should be used as the maximum distance
     *                         between the device the point defined using {@code pointLat} and
     *                         {@code pointLng}.
     * @param inArea Indicates whether or not you would like the rule to evaluated as true when the
     *               user is within an area or outside of it.
     * @return The {@code RulePredicate} that can be used for constructing a {@link Rule}.
     */
    @Nullable
    public static RulePredicate withinRange(double pointLat, double pointLng, int distanceInMeters, boolean inArea){

        try {
            String encodedBase64DataForLocation = encodePoints(pointLat, pointLng, distanceInMeters);
            UUID randomGUID = UUID.randomUUID();
            String guidAsString = randomGUID.toString().toUpperCase();

            RulePredicate predicate = equals("ctx.flybits.area.query.inArea." + guidAsString, inArea);
            predicate.addPluginData("ctx.flybits.area.query.inArea." + guidAsString, encodedBase64DataForLocation);
            return predicate;
        }catch (UnsupportedEncodingException | JSONException e){
            return null;
        }
    }

    private static String getQueryParam(String plugin, String id) throws InvalidParameterException{
        try {
            String[] result = plugin.split("\\.");
            int index = plugin.lastIndexOf(".");
            if (result.length == 4) {
                return plugin.substring(0, index) + ".query." + result[result.length - 1] + "." + id;
            }
        }catch (Exception e){}
        return null;
    }

    private static String encodePoints(double pointLat, double pointLng, int distanceInMeters) throws UnsupportedEncodingException, JSONException {
        LatLng myLatLng = new LatLng(pointLat, pointLng);

        LatLng topLeft      = computeOffset(myLatLng, distanceInMeters, 315.0);
        LatLng topRight     = computeOffset(myLatLng, distanceInMeters, 45.0);
        LatLng bottomRight  = computeOffset(myLatLng, distanceInMeters, 135.0);
        LatLng bottomLeft   = computeOffset(myLatLng, distanceInMeters, 225.0);

        JSONObject latlng1  = new JSONObject();
        latlng1.put("lat", topLeft.latitude);
        latlng1.put("lng", topLeft.longitude);

        JSONObject latlng2  = new JSONObject();
        latlng2.put("lat", topRight.latitude);
        latlng2.put("lng", topRight.longitude);

        JSONObject latlng3  = new JSONObject();
        latlng3.put("lat", bottomRight.latitude);
        latlng3.put("lng", bottomRight.longitude);

        JSONObject latlng4  = new JSONObject();
        latlng4.put("lat", bottomLeft.latitude);
        latlng4.put("lng", bottomLeft.longitude);

        JSONArray shapes    = new JSONArray();
        shapes.put(latlng1);
        shapes.put(latlng2);
        shapes.put(latlng3);
        shapes.put(latlng4);
        shapes.put(latlng1);

        JSONArray area     = new JSONArray();
        area.put(shapes);

        JSONObject multipleShapes     = new JSONObject();
        multipleShapes.put("shapes", area);

        JSONObject object   = new JSONObject();
        object.put("area", multipleShapes);

        byte[] data = object.toString().getBytes("UTF-8");
        return Base64.encodeToString(data, Base64.NO_WRAP);
    }

    private static LatLng computeOffset(LatLng from, double distance, double heading) {
        distance /= EARTH_RADIUS;
        heading = Math.toRadians(heading);
        // http://williams.best.vwh.net/avform.htm#LL
        double fromLat = Math.toRadians(from.latitude);
        double fromLng = Math.toRadians(from.longitude);
        double cosDistance = Math.cos(distance);
        double sinDistance = Math.sin(distance);
        double sinFromLat = Math.sin(fromLat);
        double cosFromLat = Math.cos(fromLat);
        double sinLat = cosDistance * sinFromLat + sinDistance * cosFromLat * Math.cos(heading);
        double dLng = Math.atan2(
                sinDistance * cosFromLat * Math.sin(heading),
                cosDistance - sinFromLat * sinLat);
        return new LatLng(Math.toDegrees(Math.asin(sinLat)), Math.toDegrees(fromLng + dLng));
    }

    /*
    End Location Predicate
     */
}
