package com.flybits.context.activities;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.api.FlybitsAPIConstants;
import com.flybits.commons.library.logging.Logger;
import com.flybits.context.library.R;

import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.HashMap;

/**
 * The {@code OAuthLoginBrowserActivity} is an {@link android.app.Activity} that is automatically
 * launched by the Context SDK when an application requests to get 3rd-Party information from the
 * user that will require the user to login.
 */
public class OAuthLoginBrowserActivity extends Activity {

    private final String _TAG                       = "OAuthLoginBrowserActivity";
    final String API_CONTEXT_OAUTH                  = "/context/oauth";

    /**
     * Indicates that the {@code OAuthLoginBrowserActivity} has encountered an error and therefore
     * cannot successfully authenticate with the Flybits Experience Engine.
     */
    public static final int RESULT_ACCESS_DENIED    = 2;

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        String provider     = null;
        Bundle bundle = getIntent().getExtras();
        if (bundle != null){
            provider    = bundle.getString("provider");
        }else{
            finish();
        }

        setContentView(R.layout.activity_web);
        WebView myWebView = (WebView) findViewById(R.id.web_flybits_browser);
        WebSettings webSettings = myWebView.getSettings();
        webSettings.setJavaScriptEnabled(true);

        myWebView.setWebViewClient(new WebViewClient(){
            @Override
            public boolean shouldOverrideUrlLoading(WebView view, String url) {

                Uri uri = isFlybitsURLBeingAccessed(OAuthLoginBrowserActivity.this, url);
                if (uri != null){

                    Logger.setTag(_TAG).d("Successfully Logged In");

                    Intent resultIntent = new Intent();
                    int resultCode = getResultCode(uri);
                    if (resultCode == RESULT_ACCESS_DENIED) {
                        resultIntent.putExtra("error", uri.getQueryParameter("error"));
                    }
                    setResult(resultCode, resultIntent);
                    finish();
                }

                view.loadUrl(url, getHeaders(OAuthLoginBrowserActivity.this));
                return true;
            }
        });

        String url = String.format(FlybitsAPIConstants.constructGatewayURL(OAuthLoginBrowserActivity.this, API_CONTEXT_OAUTH) + "?provider=%s", provider);
        myWebView.loadUrl(url, getHeaders(OAuthLoginBrowserActivity.this));
    }

    static Uri isFlybitsURLBeingAccessed(Context context, String url){
        try {
            URL urlObj = new URL(url);
            Uri uri = Uri.parse(urlObj.toURI().toString());

            if (startsWithFlybitsAPI(context, uri)) {

                if ( uri.getQueryParameter("provider") != null){
                    return uri;
                }
            }
        }catch(MalformedURLException | URISyntaxException exception){
            Logger.exception("OAuthLoginBrowserActivity.isFlybitsURLBeingAccessed", exception);
        }
        return null;
    }

    static HashMap<String, String> getHeaders(Context context){
        HashMap<String, String> listOfHeaders   = new HashMap<>();
        listOfHeaders.put("X-Authorization", SharedElements.INSTANCE.getSavedJWTToken());
        return listOfHeaders;
    }

    /**
     * Confirms that the {@code uri} provided has originated from the Flybits' Gateway server.
     *
     * @param uri The {@link Uri} that should be checked to see where it originated from.
     * @return true if the {@code uri} has in fact originated from a Flybits server, false
     * otherwise.
     */
    static boolean startsWithFlybitsAPI(Context context, Uri uri){
        return uri != null && uri.toString().startsWith(FlybitsAPIConstants.getGatewayURL(context));
    }

    /**
     * Get the result code for the provided {@code uri}. The result code indicates whether or not
     * the server successfully authenticated with the desired provider.
     *
     * @param uri The {@link Uri} that should be used to get the result code for.
     * @return The result code that indicates whether or not the request for authenticating with a
     * provider was successful.
     */
    static int getResultCode(Uri uri){
        if (uri.getQueryParameter("error") != null && !uri.getQueryParameter("error").isEmpty()) {
            return RESULT_ACCESS_DENIED;
        }
        return RESULT_OK;
    }
}
