package com.flybits.context;


import android.content.Context;
import android.content.SharedPreferences;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.FlybitsScope;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.User;
import com.flybits.context.db.ContextDatabase;
import com.flybits.context.models.ContextPriority;
import com.flybits.context.models.internal.Plugin;
import com.flybits.context.services.FlybitsContextPluginService;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static android.content.Context.MODE_PRIVATE;

/**
 * The {@code ContextScope} is the class responsible for setting all Context Options when starting
 * the Context SDK. This is an important step because it defines all the attributes needed for
 * context processing.
 */
public class ContextScope extends FlybitsScope {

    public static final String ROOT     = "/context";
    private String _TAG = "ContextScope";

    private long timeToUploadContext;
    private long timeToRetrieveRules;
    private long timeToRetrieveContextPlugins;
    private boolean autoStartContextCollection;
    private HashMap<String, String> classesAsString;

    public static ContextScope SCOPE = new ContextScope();

    ContextScope(){
        super("ContextSDK");
    }

    /**
     * Constructor that initializes the {@code ContextScope}. This constructor activates the
     * uploading Context data to the Flybits Context Server as well as the retrieval of Context
     * Rules.
     *
     * @param timeToUploadContext The time to upload Context data to the server.
     * @param unit The unit of the measure for time.
     */
    public ContextScope(long timeToUploadContext, @NonNull TimeUnit unit){

        this(timeToUploadContext, -1, unit, -1, false, null);
    }

    /**
     * Constructor that initializes the {@code ContextScope}. This constructor activates the
     * uploading Context data to the Flybits Context Server as well as the retrieval of Context
     * Rules.
     *
     * @param timeToUploadContext The time to upload Context data to the server.
     * @param unit The unit of the measure for time.
     * @param classes HashMap of custom classes that should be started when Context Plugin is
     *                enabled.
     * @deprecated Use the more optimized {@link #ContextScope(long, long, TimeUnit, HashMap)}.
     */
    @Deprecated
    public ContextScope(long timeToUploadContext, @NonNull TimeUnit unit,
                        HashMap<String, Class<? extends FlybitsContextPluginService>> classes){

        this(timeToUploadContext, -1, unit, -1, true, classes);
    }

    /**
     * Constructor that initializes the {@code ContextScope}. This constructor activates the
     * uploading Context data to the Flybits Context Server as well as the retrieval of Context
     * Rules.
     *
     * @param timeToUploadContext The time to upload Context data to the server.
     * @param unit The unit of the measure for time.
     * @param classes HashMap of custom classes that should be started when Context Plugin is
     *                enabled.
     */
    public ContextScope(long timeToUploadContext, long timeToRetrieveContextPlugins, @NonNull TimeUnit unit,
                        HashMap<String, Class<? extends FlybitsContextPluginService>> classes){

        this(timeToUploadContext, -1, unit, timeToRetrieveContextPlugins, true, classes);
    }

    /**
     * Constructor that initializes the {@code ContextScope}. This constructor only activates the
     * uploading Context data to the Flybits Context Server.
     *
     * @param timeToUploadContext The time to upload Context data to the server.
     * @param timeToRetrieveRules The time to refresh the list of Rules associated to this
     *                            application. This is an important feature as it indicates which
     *                            rules are true and which are false.
     * @param unit The unit of the measure for time.
     * @param autoStartContextCollection Should you automatically start Context Plugins.
     * @param classes HashMap of custom classes that should be started when Context Plugin is
     *                enabled.
     */
    private ContextScope(long timeToUploadContext, long timeToRetrieveRules,
                         @NonNull TimeUnit unit,long timeToRetrieveContextPlugins, boolean autoStartContextCollection,
                        HashMap<String, Class<? extends FlybitsContextPluginService>> classes){
        super("CONTEXT_SDK");

        this.autoStartContextCollection = autoStartContextCollection;
            classesAsString                 = new HashMap<>();

            if (timeToUploadContext != -1) {
                this.timeToUploadContext = unit.toMinutes(timeToUploadContext);
                if (this.timeToUploadContext == 0) {
                    this.timeToUploadContext = 1;
                }
            }else{
                this.timeToUploadContext = -1;
            }

            if (timeToRetrieveRules != -1) {
                this.timeToRetrieveRules = unit.toMinutes(timeToRetrieveRules);
                if (this.timeToRetrieveRules == 0) {
                    this.timeToRetrieveRules = 1;
            }
        }else{
            this.timeToRetrieveRules = -1;
        }

        if (timeToRetrieveContextPlugins != -1) {
            this.timeToRetrieveContextPlugins = unit.toSeconds(timeToRetrieveContextPlugins);
            if (this.timeToRetrieveContextPlugins == 0) {
                this.timeToRetrieveContextPlugins = 86400;
            }
        }else{
            this.timeToRetrieveContextPlugins = 86400;
        }

        //Make sure that classes are allowed to be null
        if (autoStartContextCollection && classes != null && classes.size() > 0){

            Iterator it = classes.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<String, Class> pair = (Map.Entry<String, Class>) it.next();
                classesAsString.put(pair.getKey(), pair.getValue().getCanonicalName());
                it.remove();
            }
        }
    }

    long getTimeToUploadContext(){return this.timeToUploadContext;}

    long getTimeToRetrieveRules(){return this.timeToRetrieveRules;}

    boolean isAutoStartContextCollection(){
        return autoStartContextCollection;
    }

    @Override
    public void onConnected(Context context, User user) {
        Logger.setTag(_TAG).d("onConnected: " + user);
        if (timeToUploadContext > 0){
            startUploadingContext(context);
        }
        if (timeToRetrieveRules > 0){
            startRetrievingRules(context);
        }
        if (autoStartContextCollection){
            ContextManager.registerForPluginUpdates(context, classesAsString, timeToRetrieveContextPlugins);
        }
    }

    @Override
    public void onDisconnected(Context context, String jwtToken) {
        Logger.setTag(_TAG).d("onDisconnected");
        ContextManager.unregisterUploadingContext(context);
        ContextManager.unregisterFromRuleCollection(context);
        ContextManager.unregisterPluginContext(context);
        clearContextPreferences(context);

        List<Plugin> listOfOldPlugins  = ContextDatabase.getDatabase(context).pluginDao().getAll();
        //Go Through all plugins previously stored on the DB
        for (Plugin plugin : listOfOldPlugins){
            ContextPluginsService.stopContextPlugin(context, classesAsString, plugin);
        }
    }

    @Override
    public void onAccountDestroyed(Context context, String jwtToken) {
        Logger.setTag(_TAG).d("onAccountDestroyed");
        onDisconnected(context, jwtToken);
    }

    void startUploadingContext(Context context){
        Logger.setTag(_TAG).d("Activating Contextual Uploading");

        SharedPreferences.Editor preferences   = ContextScope.getContextPreferences(context).edit();
        preferences.putLong(ContextUploadingService.PREF_REFRESH_RATE, TimeUnit.MINUTES.toSeconds(timeToUploadContext));
        preferences.apply();
        ContextManager.registerUploadingContext(context, ContextPriority.HIGH, timeToUploadContext, 1, TimeUnit.MINUTES);
    }

    void startRetrievingRules(Context context){
        Logger.setTag(_TAG).d("Activating Contextual Uploading");

        SharedPreferences.Editor preferences   = ContextScope.getContextPreferences(context).edit();
        preferences.putLong(ContextRulesService.PREF_RULES_REFRESH_TIME, TimeUnit.MINUTES.toSeconds(timeToRetrieveRules));
        preferences.apply();
        ContextManager.registerForRules(context, timeToRetrieveRules, 1, TimeUnit.MINUTES);
    }

    /**
     * Get the default shared preferences for the application.
     *
     * @param context The context of the application.
     * @return The default SharedPreferences object for the application.
     */
    public static SharedPreferences getContextPreferences(Context context){
        return context.getSharedPreferences("FLYBITS_PREF_CONTEXT", MODE_PRIVATE );
    }

    private void clearContextPreferences(Context context){
        SharedPreferences.Editor preferences   = getContextPreferences(context).edit();
        preferences.clear();
        preferences.apply();
    }

}