package com.flybits.context.plugins.carrier;

import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;

import com.flybits.context.models.ContextData;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * The {@code CarrierData} that contains information about the SIM card and the SIM carrier at any
 * given time.
 */
public class CarrierData extends ContextData implements Parcelable {

    private String mcc;
    private String mnc;

    /**
     * Default constructor needed for generics instantiation
     */
    public CarrierData(){}

    /**
     * Constructor used for un-flattening a {@code CarrierData} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code CarrierData} parcel.
     */
    protected CarrierData(Parcel in){
        super();
        this.mcc = in.readString();
        this.mnc = in.readString();
        setTime(in.readLong());
    }

    /**
     * Constructor used to define the {@code CarrierData} object based on a preset String that
     * indicates the {@code MCC} and {@code MNC} codes.
     *
     * @param simOperator The conjunction of the operator information in the following format:
     *                    XXXYYY where XXX is the MCC code and YYY is the MNC code.
     */
    public CarrierData(String simOperator){
        super();
        this.mcc    = getSIM_MCC(simOperator);
        this.mnc    = getSIM_MNC(simOperator);
    }

    /**
     * Constructor used to define the {@code CarrierData} object based on a preset String that
     * indicates the {@code MCC} and {@code MNC} codes.
     *
     * @param mcc The MCC code which identifies the country of origin for the SIM operator.
     * @param mnc The MNC code which identifies the specific operator.
     */
    public CarrierData(String mcc, String mnc){
        super();
        this.mcc    = mcc;
        this.mnc    = mnc;
    }

    /**
     * The MCC code which represents the country of origin for the SIM operator.
     *
     * @return The 3-letter String representation of the MCC code.
     */
    public String getMcc() {
        return mcc;
    }


    /**
     * The MNC code which represents a specific SIM operator.
     *
     * @return The 3-letter String representation of the MNC code.
     */
    public String getMnc() {
        return mnc;
    }

    @Override
    public String toJson() {

        JSONObject object=new JSONObject();
        try {
            if (mcc != null) {
                object.put("mcc", mcc);
            }

            if (mnc != null) {
                object.put("mnc", mnc);
            }
        }catch (JSONException exception){}

        return object.toString();
    }

    @Override
    public void fromJson(String json) {
        try {
            JSONObject jsonObj  = new JSONObject(json);
            if (!jsonObj.isNull("mcc")) {
                mcc = jsonObj.getString("mcc");
            }
            if (!jsonObj.isNull("mnc")) {
                mnc = jsonObj.getString("mnc");
            }

        }catch (JSONException exception){}
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        CarrierData that = (CarrierData) o;

        if (!mcc.equals(that.mcc)) return false;
        return mnc.equals(that.mnc);
    }

    @Override
    public String getPluginID() {
        return "ctx.sdk.carrier";
    }

    @Override
    public String toString(){
        return toJson();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(mcc);
        dest.writeString(mnc);
        dest.writeLong(getTime());
    }

    /**
     * Parcelable.Creator that instantiates {@code CarrierData} objects
     */
    public static final Creator<CarrierData> CREATOR = new Creator<CarrierData>() {
        public CarrierData createFromParcel(Parcel in) {
            return new CarrierData(in);
        }

        public CarrierData[] newArray(int size) {
            return new CarrierData[size];
        }
    };

    String getSIM_MNC(String simOperator){

        if (!TextUtils.isEmpty(simOperator)) {
            try{
                return String.valueOf(simOperator.substring(3));
            }catch (StringIndexOutOfBoundsException e){
                return "";
            }
        }
        return "";
    }

    String getSIM_MCC(String simOperator){

        if (!TextUtils.isEmpty(simOperator)) {
            try{
                return String.valueOf(simOperator.substring(0, 3));
            }catch (StringIndexOutOfBoundsException e){
                return "";
            }
        }
        return "";
    }
}
