package com.flybits.context.plugins;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.FlybitsAPIConstants;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.context.ContextScope;
import com.flybits.context.activities.OAuthLoginBrowserActivity;
import com.flybits.context.plugins.oauth.AccessToken;
import com.flybits.context.plugins.oauth.IFlybitsOAuthCallback;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * The {@code OAuthContextPlugin} can be used to define a {@link ContextPlugin} that needs to be a
 * success user login using the OAuth authentication method. The {@code OAuthContextPlugin}
 * abstracts most of the features in order to simply the activation process for the application. If
 * a {@link ContextPlugin} would like to retrieve information about a user's Facebook, Twitter,
 * Spotify accounts, then the {@code OAuthContextPlugin} should be used. The
 * {@code OAuthContextPlugin} will automatically the user to login whenever the Contextual
 * information is needed to be processed. In most cases, the login credentials are only needed to be
 * entered once as the connection will persist.
 */
public class OAuthContextPlugin implements ContextPlugin {

    public static final String API_CONTEXT_OAUTH                = ContextScope.ROOT+"/oauth";
    public static final String API_CONTEXT_OAUTH_ACCESSTOKEN    = API_CONTEXT_OAUTH + "/accesstoken";
    public static final String API_CONTEXT_OAUTH_USERS          = API_CONTEXT_OAUTH + "/users";
    private final String _TAG   = "OAuthContextPlugin";

    private String provider;
    private String accessToken;
    private Long expiresAt;
    private IFlybitsOAuthCallback callback;
    private Activity activity;
    private int browserRequestCode;

    private OAuthContextPlugin(Builder builder) {
        this.provider           = builder.provider;
        this.accessToken        = builder.accessToken;
        this.expiresAt          = builder.expiresAt;
        this.activity           = builder.activity;
        this.browserRequestCode = builder.browserRequestCode;
        this.callback           = builder.callback;
    }

    /**
     * Get the {@code AccessToken} associated to the provider of this {@code OAuthContextPlugin}.
     * The provider indicates which platform the user's has logged into such as Facebook, Twitter,
     * Spotify, etc. Once the user has successfully logged into the platform an {@code AccessToken}
     * will be provided from the Provider.
     *
     * @return The unique {@code AccessToken} that represents a user's session for a particular
     * OAuth provider.
     */
    public String getAccessToken() {
        return accessToken;
    }

    /**
     * Get the Activity that initiated the retrieval of OAuth data from a Flybits-enabled WebView.
     * This is needed is the Application should like to know whether or not the OAuth request was
     * successful or not.
     *
     * @return The {@link Activity} that represented the initiating {@link Activity} that requested
     * for OAuth information to be retrieved from.
     */
    public Activity getActivity() {
        return activity;
    }

    /**
     * Get the Activity request code for the activity that started the {@code OAuthContextPlugin},
     * this is used when returning the result from the {@code OAuthContextPlugin} Activity.
     *
     * @return the request code that represents the activity that started the
     * {@code OAuthContextPlugin}.
     */
    public int getBrowserRequestCode()
    {
        return browserRequestCode;
    }

    /**
     * Callback that is used to indicate whether the OAuth callback was successful or not.
     *
     * @return The callback for {@link IFlybitsOAuthCallback}.
     */
    public IFlybitsOAuthCallback getCallback() {
        return callback;
    }

    /**
     * Get the expiry time as epoch of the {@code AccessToken}.
     *
     * @return The epoch time of the expiry of the {@code AccessToken}.
     */
    public Long getExpiresAt() {
        return expiresAt;
    }

    /**
     * Get the {@code provider} of the {@code AccessToken} that should allow the user to sign-in
     * using the {@code OAuthContextPlugin}.
     *
     * @return The {@code provider} that the {@code OAuthContextPlugin} should display to the user
     * for sign-in to get the Contextual information about the user.
     */
    public String getProvider() {
        return provider;
    }

    @Override
    public void onStart(final Context context) {
        if(getAccessToken() == null) {

            String provider = getProvider();
            Intent intent = new Intent(context, OAuthLoginBrowserActivity.class);
            intent.putExtra("provider", provider);
            getActivity().startActivityForResult(intent, getBrowserRequestCode());

        } else {
            final Handler handler = new Handler(Looper.getMainLooper());
            final ExecutorService executorService = Executors.newSingleThreadExecutor();
            executorService.execute(new Runnable() {
                public void run() {
                    try {
                        AccessToken token = new AccessToken(provider, accessToken, expiresAt);
                        Result result = FlyAway.post(context, API_CONTEXT_OAUTH_ACCESSTOKEN, token.toJson(), null, "OAuthContextPlugin.onStart", null);
                        if (result.getStatus()   == RequestStatus.COMPLETED && getCallback() != null){
                            handler.post(new Runnable() {
                                @Override
                                public void run() {
                                    getCallback().onOAuthLoginComplete(true, getAccessToken());
                                }
                            });
                        }else if (getCallback() != null){
                            handler.post(new Runnable() {
                                @Override
                                public void run() {
                                    getCallback().onOAuthLoginComplete(false, null);
                                }
                            });
                        }
                    }catch (FlybitsException e){
                        Logger.exception("OAuthContextPlugin.register", e);
                    }
                }
            });
        }

        Logger.setTag(_TAG).d("Activated OAuth Plugin: " + getProvider());
    }

    @Override
    public void onStop(final Context context) {
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        executorService.execute(new Runnable() {
            public void run() {

                try {
                    String route    = FlybitsAPIConstants.constructGatewayURL(context, API_CONTEXT_OAUTH_USERS);
                    String url      = String.format(route + "?provider=%s", provider);
                    Result result   = FlyAway.delete(context, url,  "OAuthContextPlugin.onStop", null);
                    if (result.getStatus()   == RequestStatus.COMPLETED &&  getCallback() != null){
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                getCallback().onOAuthDeleteComplete(true);
                            }
                        });
                        Logger.setTag(_TAG).d("Deactivated: " + getProvider());
                    }else if ( getCallback() != null){
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                getCallback().onOAuthDeleteComplete(false);
                            }
                        });
                    }
                }catch (FlybitsException e){
                    Logger.exception("OAuthContextPlugin.unregister", e);
                }
            }
        });

        Logger.setTag(_TAG).d("Deactivated OAuth Plugin: " + getProvider());
    }

    @Override
    public void onRefresh(Context context) {
        //You currently cannot refresh the OAuth Context Plugin
    }

    /**
     * This {@code Builder} class is responsible for allowing the application to register an
     * OAuth-based Context Plugin within its application.
     */
    public static class Builder {

        private String accessToken;
        private long expiresAt;
        private String provider;
        private Activity activity;
        private int browserRequestCode;
        private IFlybitsOAuthCallback callback;

        /**
         * Constructor used to define the critical attributes that the {@link OAuthContextPlugin}
         * needs in order to operate properly.
         *
         * @param provider The 3rd party provider such as Facebook, Twitter, Spotify, etc. to which
         *                 the {@link OAuthContextPlugin}.
         * @param activity The {@link Activity} that initiated the {@link OAuthContextPlugin}.
         * @param requestCode The {@code requestCode} that represents the {@code Activity} that
         *                    started the {@link OAuthContextPlugin}.
         */
        public Builder(@NonNull String provider, @NonNull Activity activity, int requestCode){
            this.provider = provider;
            this.activity = activity;
            this.browserRequestCode = requestCode;
        }

        /**
         * Used to create the {@link OAuthContextPlugin} based on the attributes set within the
         * {@code Builder}.
         *
         * @return The {@link OAuthContextPlugin} object that can registered through the
         * {@link ContextPlugin#onStart(Context)}.
         */
        public OAuthContextPlugin build(){
            return new OAuthContextPlugin(this);
        }

        /**
         * Set the {@code AccessToken} associated to the provider of this {@code OAuthContextPlugin}.
         * The provider indicates which platform the user's has logged into such as Facebook, Twitter,
         * Spotify, etc. Once the user has successfully logged into the platform an {@code AccessToken}
         * will be provided from the Provider.
         *
         * @param accessToken The {@code accessToken} received from the 3rd party using the OAuth
         *                    mechanism for user retrieval.
         * @return The {@code Builder} class which can be reused to add additional attributes to.
         */
        public Builder setAccessToken(@NonNull String accessToken) {
            return setAccessToken(accessToken, -1);
        }

        /**
         * Set the {@code AccessToken} associated to the provider of this {@code OAuthContextPlugin}.
         * The provider indicates which platform the user's has logged into such as Facebook, Twitter,
         * Spotify, etc. Once the user has successfully logged into the platform an {@code AccessToken}
         * will be provided from the Provider.
         *
         * @param accessToken The {@code accessToken} received from the 3rd party using the OAuth
         *                    mechanism for user retrieval.
         * @param expiresAt Indicates when the {@code accessToken} expires and a new should be
         *                  requested for.
         * @return The {@code Builder} class which can be reused to add additional attributes to.
         */
        public Builder setAccessToken(@NonNull String accessToken, long expiresAt) {
            this.accessToken  = accessToken;
            this.expiresAt = expiresAt;
            return this;
        }

        /**
         * Sets the {@code IFlybitsOAuthCallback}
         *
         * @param callback The callback that indicates whether or not the OAuth requests are
         *                 successful or not.
         * @return The {@code Builder} class which can be reused to add additional attributes to.
         */
        public Builder setOAuthBrowserCallback(@NonNull IFlybitsOAuthCallback callback) {
            this.callback = callback;
            return this;
        }
    }
}
