package com.flybits.context.plugins.location


import android.Manifest
import android.content.Context
import android.os.Bundle
import androidx.work.WorkerParameters
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.context.models.ContextData
import com.flybits.context.plugins.FlybitsContextPlugin
import com.flybits.context.services.FlybitsContextPluginsWorker
import com.google.android.gms.location.FusedLocationProviderClient
import com.google.android.gms.location.LocationServices
import com.google.android.gms.tasks.Tasks
import java.util.concurrent.ExecutionException

const val TAG = "PluginLocation"

/**
 * The [LocationContextPluginWorker] is a worker responsible for retrieving the last location.
 * This will be triggered by `WorkManager` on start of location plugin to fetch the last location after specified interval.
 * The minimum interval for the worker to execute is 900 seconds.
 * Permissions required are : ACCESS_FINE_LOCATION and ACCESS_COARSE_LOCATION.
 */
class LocationContextPluginWorker(context: Context, workerParameters: WorkerParameters) :
    FlybitsContextPluginsWorker(context, workerParameters) {

    private var refreshTime: Long = 0
    private lateinit var fusedLocationClient: FusedLocationProviderClient

    override fun initialize(bundle: Bundle?) {
        refreshTime = inputData.getLong(FlybitsContextPlugin.EXTRA_MINIMUM_REFRESH_TIME, 60)
        fusedLocationClient = LocationServices.getFusedLocationProviderClient(applicationContext)
    }

    override fun isSupported(): Boolean {
        return true
    }

    override fun getData(): ContextData {
        Logger.setTag(TAG).d("Fetching Location Data ...")
        try {
            // Using Tasks.await to get the result synchronously without callback.
            val lastLocation = Tasks.await(fusedLocationClient.lastLocation)
            return if (lastLocation != null) {
                LocationData(lastLocation)
            } else {
                throw FlybitsException("Issue retrieving location values.")
            }
        } catch (e: ExecutionException) {
            throw Exception(e.message)
        } catch (e: InterruptedException) {
            throw Exception(e.message)
        }
    }

    override fun getRequiredPermissions(): Array<String> {
        return arrayOf(
            Manifest.permission.ACCESS_FINE_LOCATION,
            Manifest.permission.ACCESS_COARSE_LOCATION
        )
    }

}