package com.flybits.context.plugins.location;

import android.Manifest;
import android.content.pm.PackageManager;
import android.location.Location;
import android.os.Build;
import android.os.Bundle;
import android.support.v4.content.ContextCompat;

import com.flybits.commons.library.logging.Logger;
import com.flybits.context.models.ContextData;
import com.flybits.context.services.FlybitsContextPluginService;
import com.google.android.gms.location.FusedLocationProviderClient;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.tasks.Tasks;

import java.util.concurrent.ExecutionException;

/**
 * The {@link LocationContextPluginService} is a service responsible for retrieving the last location.
 * The minimum interval for the service to execute is specified by refresh time.
 * Permissions required are : ACCESS_FINE_LOCATION and ACCESS_COARSE_LOCATION.
 */
public class LocationContextPluginService extends FlybitsContextPluginService {
    private final static String _TAG = "PluginLocation";
    private FusedLocationProviderClient fusedLocationProviderClient;
    private long minDisplacement = -1;

    @Override
    public void initialize(Bundle bundle) {
        fusedLocationProviderClient = LocationServices.getFusedLocationProviderClient(getApplicationContext());
        if (bundle != null && bundle.containsKey("minDisplacement")) {
            minDisplacement = bundle.getLong("minDisplacement");
        }
    }

    @Override
    public boolean isSupported() {
        return Build.VERSION.SDK_INT < 23 ||
                ContextCompat.checkSelfPermission(getBaseContext(), Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED;
    }

    @Override
    public ContextData getData() {
        Logger.setTag(_TAG).d("Fetching Data ...");
        try {
            // Using Tasks.await to get the result synchronously without callback.
            if (isSupported()) {
                Location lastLocation = Tasks.await(fusedLocationProviderClient.getLastLocation());
                return new LocationData(lastLocation);
            } else {
                return null;
            }
        } catch (ExecutionException | InterruptedException e) {
            Logger.setTag(_TAG).e("Exception : " + e.getMessage());
        }
        return null;
    }


    @Override
    public void onDestroy() {
        Logger.setTag(_TAG).d("...Destroyed");
        super.onDestroy();
    }

    @Override
    public String[] getRequiredPermissions() {
        return new String[]{Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION};
    }
}
