package com.flybits.context.plugins.fitness;

import android.content.Context;

import com.flybits.commons.library.logging.Logger;
import com.google.android.gms.auth.api.signin.GoogleSignIn;
import com.google.android.gms.fitness.Fitness;
import com.google.android.gms.fitness.data.Bucket;
import com.google.android.gms.fitness.data.DataPoint;
import com.google.android.gms.fitness.data.DataSet;
import com.google.android.gms.fitness.data.DataType;
import com.google.android.gms.fitness.data.Field;
import com.google.android.gms.fitness.request.DataReadRequest;
import com.google.android.gms.fitness.result.DataReadResponse;

import java.util.Calendar;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * Class Responsible for Retrieving the Steps for Fitness Plugin
 */
class FitnessUtils {

    static final String _TAG = "FitnessData";
    private Context _context;

    /**
     * Constructor to assign the context.
     *
     * @param context Context associated with the application.
     */
    FitnessUtils(Context context) {
        _context = context;
    }


    /**
     * Construct time interval of which fitness data will be calculated
     *
     * @param callback The callback used to indicate caller about the fitness data.
     */
    void getStepsToday(IFitnessData callback) {
        Calendar calStart = Calendar.getInstance();
        Calendar calEnd = Calendar.getInstance();
        Date now = new Date();
        calStart.setTime(now);

        calStart.set(Calendar.HOUR_OF_DAY, 0);
        calStart.set(Calendar.MINUTE, 0);
        calStart.set(Calendar.SECOND, 1);

        getSteps(calStart.getTimeInMillis(), calEnd.getTimeInMillis(), callback);
    }


    /**
     * Request the Fitness Api for the count of steps
     *
     * @param startTime The start time from where the fitness data will be calculated.
     * @param endTime   The end time till where the fitness data will be calculated.
     * @param callback  The callback used to indicate caller about the fitness data.
     */
    private void getSteps(long startTime, long endTime, final IFitnessData callback) {

        DataReadRequest readRequest = new DataReadRequest.Builder()
                // The data request can specify multiple data types to return, effectively
                // combining multiple data queries into one call.
                // In this example, it's very unlikely that the request is for several hundred
                // datapoints each consisting of a few steps and a timestamp.  The more likely
                // scenario is wanting to see how many steps were walked per day, for 7 days.
                .aggregate(DataType.TYPE_STEP_COUNT_DELTA, DataType.AGGREGATE_STEP_COUNT_DELTA)
                // Analogous to a "Group By" in SQL, defines how data should be aggregated.
                // bucketByTime allows for a time span, whereas bucketBySession would allow
                // bucketing by "sessions", which would need to be defined in code.
                .bucketByTime(1, TimeUnit.DAYS)
                .setTimeRange(startTime, endTime, TimeUnit.MILLISECONDS)
                .build();


        Fitness.getHistoryClient(_context, GoogleSignIn.getLastSignedInAccount(_context))
                .readData(readRequest)
                .addOnSuccessListener(dataReadResponse -> {
                    int steps = getSteps(dataReadResponse);
                    callback.onGetSteps(steps);
                })
                .addOnFailureListener(e -> callback.onCancel())
                .addOnCompleteListener(task -> {
                });
    }


    /**
     * Returns the actual steps data of the authenticated user
     *
     * @param dataReadResult The {@link DataReadResponse} retrieved from calling getHistoryClient()
     */
    private int getSteps(DataReadResponse dataReadResult) {
        int addition = 0;

        if (dataReadResult.getBuckets() == null || dataReadResult.getBuckets().size() == 0) {
            return -1; //cancel
        }

        for (Bucket bucket : dataReadResult.getBuckets()) {

            if (bucket.getDataSets().size() == 0) {
                return -1; //cancel
            }

            for (DataSet dataSet : bucket.getDataSets()) {

                if (dataSet.getDataPoints().size() == 0) {
                    return -1; //cancel
                }

                for (DataPoint point : dataSet.getDataPoints()) {

                    try {
                        addition += point.getValue(Field.FIELD_STEPS).asInt();
                    } catch (Exception e) {
                        Logger.exception("FitnessUtils.getSteps", e);
                    }
                }
            }
        }

        Logger.setTag(_TAG).d("Number Of Steps: " + addition);
        return addition;
    }

}
