package com.flybits.context.plugins

import android.app.ActivityManager
import android.app.Notification
import android.arch.persistence.room.ColumnInfo
import android.arch.persistence.room.Entity
import android.arch.persistence.room.Ignore
import android.arch.persistence.room.PrimaryKey
import android.content.Context
import android.content.Intent
import android.content.SharedPreferences
import android.os.Bundle
import android.support.v4.content.ContextCompat
import androidx.work.*
import com.flybits.commons.library.logging.Logger
import com.flybits.context.ContextScope
import com.flybits.context.ReservedContextPlugin
import com.flybits.context.db.ContextDatabase
import com.flybits.context.exceptions.InvalidContextPluginException
import com.flybits.context.plugins.activity.ActivityContextPluginService
import com.flybits.context.plugins.battery.BatteryContextPluginService
import com.flybits.context.plugins.battery.BatteryContextPluginWorker
import com.flybits.context.plugins.beacon.BeaconScanningService
import com.flybits.context.plugins.carrier.CarrierContextPluginService
import com.flybits.context.plugins.carrier.CarrierContextPluginWorker
import com.flybits.context.plugins.fitness.FitnessContextPluginService
import com.flybits.context.plugins.fitness.FitnessContextPluginWorker
import com.flybits.context.plugins.language.LanguageContextPluginService
import com.flybits.context.plugins.language.LanguageContextPluginWorker
import com.flybits.context.plugins.location.LocationContextPluginService
import com.flybits.context.plugins.location.LocationContextPluginWorker
import com.flybits.context.plugins.network.NetworkContextPluginService
import com.flybits.context.plugins.network.NetworkContextPluginWorker
import com.flybits.context.plugins.weather.WeatherContextPluginService
import com.flybits.context.plugins.weather.WeatherContextPluginWorker
import com.flybits.context.services.FlybitsContextPluginService
import com.flybits.context.services.FlybitsContextPluginsWorker
import com.flybits.context.utils.ContextUtilities
import com.google.android.gms.gcm.GcmNetworkManager
import com.google.android.gms.gcm.PeriodicTask
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.util.concurrent.TimeUnit
import java.util.concurrent.TimeUnit.*

/**
 * Represents a Plugin.
 */
@Entity(tableName = FlybitsContextPlugin.TABLE)
class FlybitsContextPlugin : ContextPlugin {

    /**
     * Get the allowed flex time that indicates the maximum that a [FlybitsContextPlugin] can
     * exceed its [.getRefreshTime] before forcing itself to refresh.
     *
     * @return The flexible refresh time for retrieving the Context Data of this
     * [FlybitsContextPlugin] in seconds.
     */
    @ColumnInfo(name = COLUMN_REFRESH_TIME_FLEX)
    var refreshTimeFlex: Long = 0
        private set

    /**
     * Get the refresh timer when this [FlybitsContextPlugin] should refresh its Context Data.
     *
     * @return The refresh rate for retrieving the Context Data of this [FlybitsContextPlugin]
     * in seconds.
     */
    @ColumnInfo(name = COLUMN_REFRESH_TIME)
    var refreshTime: Long = 0
        private set

    /**
     * @return Whether or not the service associated with this [FlybitsContextPlugin] is currently running;
     */
    @ColumnInfo(name = COLUMN_IS_RUNNING)
    var isRunning: Boolean = false
        private set

    @Ignore
    private var foregroundServiceNotification: Notification? = null

    @Ignore
    private var extras: Bundle? = null

    @Ignore
    private var contextDatabase: ContextDatabase? = null

    /**
     * Get the [FlybitsContextPluginService] or [FlybitsContextPluginsWorker] that is associated to this
     * [FlybitsContextPlugin] and should be periodically executed.
     *
     * @return The [FlybitsContextPluginService] that should be periodically executed.
     */
    @PrimaryKey
    @ColumnInfo(name = COLUMN_SERVICE)
    var contextPluginRetriever: Class<*>
        private set

    /**
     * Determines if the application developer has indicated that the refresh time is less than 60
     * seconds if this is the case, then a custom service should be run, otherwise a
     * `GcmTaskService` will be used.
     *
     * @return true if a custom service should run based on a custom timer, false will indicate that
     * the [.GcmTaskService] will be used.
     *
     * Please Note if using [FlybitsContextPluginsWorker] the minimum refresh time is 900 seconds.
     */
    private val isCustomRefreshSet: Boolean
        get() = refreshTime < 60


    /**
     * Constructor used to initialize [FlybitsContextPlugin].
     *
     * @param refreshTimeFlex The refreshTimeFlex is interval rate to execute the retrieving the Context Data by default in Seconds.
     * @param refreshTime The refresh rate for retrieving the Context Data of this [FlybitsContextPlugin] by default in Seconds.
     * @param isRunning Whether or not the service associated with this FlybitsContextPlugin is currently running.
     * @param contextPluginRetriever The [FlybitsContextPluginService] or [FlybitsContextPluginsWorker] that is associated to this [FlybitsContextPlugin].
     */
    constructor(
        refreshTimeFlex: Long,
        refreshTime: Long,
        isRunning: Boolean,
        contextPluginRetriever: Class<*>
    ) {
        this.refreshTimeFlex = refreshTimeFlex
        this.refreshTime = refreshTime
        this.isRunning = isRunning
        this.contextPluginRetriever = contextPluginRetriever
    }

    /**
     * Constructor used to initialize [FlybitsContextPlugin].
     *
     * @param builder The [FlybitsContextPlugin.Builder] where all the attributes about the
     * [FlybitsContextPlugin] are initialized.
     */
    private constructor(builder: Builder) {
        if (builder.extras != null) {
            extras = builder.extras
        }

        contextPluginRetriever = if (builder.isWorker) {
            builder.taskWorker as Class<out FlybitsContextPluginsWorker>
        } else {
            builder.taskService as Class<out FlybitsContextPluginService>
        }

        refreshTimeFlex = builder.timeInSecondsFlex
        refreshTime = builder.timeInSeconds
        foregroundServiceNotification = builder.foregroundServiceNotification

        if (refreshTime != builder.GRETZKYS_UNIQUE_ID.toLong() && refreshTime < 60 && foregroundServiceNotification == null) {
            throw IllegalStateException("foregroundServiceNotification cannot be null if refresh time is less than 60 seconds. Use Builder.setForegroundServiceNotification().")
        }
    }

    override fun equals(other: Any?): Boolean {
        if (other !is FlybitsContextPlugin) {
            return false
        }

        val another = other as FlybitsContextPlugin?
        return if (another?.contextPluginRetriever?.superclass == FlybitsContextPluginsWorker::class.java) {
            another.contextPluginRetriever.simpleName == getWorker()?.simpleName
        } else {
            another?.contextPluginRetriever?.simpleName == getService()?.simpleName
        }
    }

    /**
     * Get the additional attributes associated to this [FlybitsContextPlugin].
     *
     * @return The additional attributes bundled within a [Bundle] object that contains all
     * the additional attributes.
     */
    private fun getExtras(): Bundle {
        return extras ?: Bundle.EMPTY
    }


    /**
     * Get the [FlybitsContextPluginsWorker] that is associated to this
     * [FlybitsContextPlugin] and should be periodically executed.
     *
     * @return The FlybitsContextPluginService] that should be periodically executed.
     */
    fun getWorker(): Class<out FlybitsContextPluginsWorker>? {
        return if (isWorkerClass(contextPluginRetriever))
            contextPluginRetriever as Class<out FlybitsContextPluginsWorker>
        else
            null
    }

    /**
     * Get the [FlybitsContextPluginService] that is associated to this
     * [FlybitsContextPlugin] and should be periodically executed.
     *
     * @return The FlybitsContextPluginService] that should be periodically executed.
     */
    @Deprecated(
        "Using this function to get the Service for a Plugin is Deprecated, " +
                "Instead use Worker for the periodic task." +
                "Replace with getWorker() to get the [FlybitsContextPluginsWorker] for plugin [FlybitsContextPlugin].",
        ReplaceWith("getWorker()")
    )
    fun getService(): Class<out FlybitsContextPluginService>? {
        return if (isServiceClass(contextPluginRetriever))
            contextPluginRetriever as Class<out FlybitsContextPluginService>
        else
            null
    }

    override fun onStart(context: Context) {
        if (isWorkerClass(contextPluginRetriever)) {
            scheduleWorkers(context)
        } else {
            scheduleService(context)
        }
    }

    override fun onStop(context: Context) {
        Logger.setTag("FlybitsContextPlugin")
            .d("onStop(): " + contextPluginRetriever.name.toString())
        if (isServiceClass(contextPluginRetriever)) {
            cancelService(context)
        } else if (isWorkerClass(contextPluginRetriever)) {
            cancelWorkByName(contextPluginRetriever.simpleName)
        }
    }

    override fun onRefresh(context: Context) {
        if (isWorkerClass(contextPluginRetriever)) {
            scheduleOneTimeWorkers()
        }
    }

    /**
     *  Start the service assigned for the [FlybitsContextPlugin].
     *  [GcmNetworkManager] builds the service to run every time after interval of [refreshTime] and
     *  with flex time [refreshTimeFlex] at the end of every interval.
     *
     *  @param context The context of [FlybitsContextPlugin]
     */
    private fun scheduleService(context: Context) {
        if (isCustomRefreshSet) {
            if (!isServiceRunning(
                    context,
                    getService() as Class<out FlybitsContextPluginService>
                ) && foregroundServiceNotification != null
            ) {
                val intent = Intent(context, getService())
                val extras = setBundle()
                extras.putParcelable(EXTRA_NOTIFICATION, foregroundServiceNotification)
                intent.putExtras(extras)
                ContextCompat.startForegroundService(context, intent)
                isRunning = true
            }
        } else {
            val preferences: SharedPreferences.Editor =
                ContextScope.getContextPreferences(context).edit()
            preferences.putLong(
                getPrefName(contextPluginRetriever),
                refreshTime
            )
            preferences.apply()

            val mGcmNetworkManager = GcmNetworkManager.getInstance(context)
            val task = PeriodicTask.Builder()
                .setTag(getService()?.simpleName)
                .setExtras(setBundle())
                .setPeriod(refreshTime)
                .setUpdateCurrent(true)
                .setPersisted(true)
                .setFlex(refreshTimeFlex)
                .setRequiredNetwork(PeriodicTask.NETWORK_STATE_ANY)
                .setService(getService())
            mGcmNetworkManager.schedule(task.build())
            isRunning = true
        }

    }


    /**
     *  Stop the service [FlybitsContextPluginService] assigned for the [FlybitsContextPlugin].
     *  [GcmNetworkManager] stops the service.
     *
     *  @param context The context of [FlybitsContextPlugin]
     */
    private fun cancelService(context: Context) {
        if (isCustomRefreshSet) {
            if (isServiceRunning(
                    context,
                    getService()
                )
            ) {
                val intent = Intent(context, getService())
                context.stopService(intent)
            }
        } else {
            val intent = Intent(context, getService())
            if (ContextUtilities.isServiceDefined(context, intent)) {
                val mGcmNetworkManager = GcmNetworkManager.getInstance(context)
                try {
                    mGcmNetworkManager.cancelTask(
                        getService()?.simpleName,
                        getService()
                    )
                } catch (ex: IllegalArgumentException) {
                    Logger.exception("FlybitsContextPlugin.stop", ex)
                }
            }
        }
        isRunning = false
    }


    /**
     *  Checks if the service assigned for the [FlybitsContextPlugin] is currently scheduled to execute.
     */
    private fun isServiceRunning(
        mContext: Context,
        serviceClass: Class<out FlybitsContextPluginService>?
    ): Boolean {
        val activityManager = mContext.getSystemService(Context.ACTIVITY_SERVICE)
        if (activityManager is ActivityManager) {
            // Loop through the running services
            for (service in activityManager.getRunningServices(Integer.MAX_VALUE)) {
                if (serviceClass?.name == service.service.className) {
                    // If the service is running then return true
                    Logger.setTag(_TAG).d("unregisterUploadingContext Completed")
                    Logger.setTag(_TAG)
                        .d("isServiceRunning: " + service.service.className + ", true")
                    return true
                }
            }
        }
        Logger.setTag(_TAG).d("isServiceRunning: " + serviceClass?.name + ", false")
        return false
    }

    /**
     *  Start the worker assigned for the [FlybitsContextPlugin].
     *  [PeriodicWorkRequest] builds the worker to run every time after interval of [refreshTime] and
     *  with flex time [refreshTimeFlex] at the end of every interval.
     *
     *  @param context The context of [FlybitsContextPlugin]
     *  Please Note : Minimum interval for worker is 900 seconds and minimum flex interval is 300 seconds.
     */
    private fun scheduleWorkers(context: Context) {
        // Cancel the Same Plugin Service Running to replace it with the worker.
        CoroutineScope(Dispatchers.Default).launch {
            cancelPluginServiceIfExists(context, getWorker())
        }

        // Worker Request Builder
        getWorker()?.let { worker ->
            val workRequest = PeriodicWorkRequest
                .Builder(worker, refreshTime, SECONDS, refreshTimeFlex, SECONDS)
                .addTag(TAG_WORK)
                .build()
            WorkManager.getInstance().enqueueUniquePeriodicWork(
                worker.simpleName
                , ExistingPeriodicWorkPolicy.REPLACE
                , workRequest
            )
            isRunning = true
        } ?: Logger.setTag(_TAG).e("Error Scheduling the Worker Request")
    }

    /**
     *  Start the worker assigned for the [FlybitsContextPlugin].
     *  [OneTimeWorkRequest] builds the worker to run only once.
     *  Will be called by [onRefresh].
     */
    private fun scheduleOneTimeWorkers() {
        //One time Worker Request Builder
        getWorker()?.let { worker ->
            val workRequest = OneTimeWorkRequest
                .Builder(worker)
                .addTag(TAG_WORK)
                .build()

            WorkManager.getInstance().enqueueUniqueWork(
                worker.simpleName
                , ExistingWorkPolicy.REPLACE
                , workRequest
            )
            isRunning = true
        } ?: Logger.setTag(_TAG).e("Error Scheduling the Worker Request")
    }


    /**
     * Cancel the Service [FlybitsContextPluginService] and delete it from table, assigned to this plugin [FlybitsContextPlugin]
     * if a worker [FlybitsContextPluginsWorker] is assigned for this plugin.
     *
     * @param context The context of [FlybitsContextPlugin]
     * @param worker The [FlybitsContextPluginsWorker] which starts the respective Plugin
     */
    internal fun cancelPluginServiceIfExists(
        context: Context,
        worker: Class<out FlybitsContextPluginsWorker>?
    ) {
        val builderService = StringBuilder()
        contextDatabase = ContextDatabase.getDatabase(context)
        builderService.append("com.flybits.context.plugins.")
        var servicePath = ""
        val isValidWorker: Boolean
        isValidWorker = when (worker) {
            BatteryContextPluginWorker::class.java -> {
                servicePath = BatteryContextPluginService::class.java.name
                true
            }
            LocationContextPluginWorker::class.java -> {
                servicePath = LocationContextPluginService::class.java.name
                true
            }
            NetworkContextPluginWorker::class.java -> {
                servicePath = NetworkContextPluginService::class.java.name
                true
            }
            LanguageContextPluginWorker::class.java -> {
                servicePath = LanguageContextPluginService::class.java.name
                true
            }
            CarrierContextPluginWorker::class.java -> {
                servicePath = CarrierContextPluginService::class.java.name
                true
            }
            FitnessContextPluginWorker::class.java -> {
                servicePath = FitnessContextPluginService::class.java.name
                true
            }
            WeatherContextPluginWorker::class.java -> {
                servicePath = WeatherContextPluginService::class.java.name
                true
            }
            else -> false
        }
        if (isValidWorker) {
            val flybitsContextPlugin =
                contextDatabase?.flybitsContextPluginDAO()
                    ?.getSingle(servicePath)
            if (flybitsContextPlugin != null) {
                flybitsContextPlugin.cancelService(context)
                contextDatabase?.flybitsContextPluginDAO()
                    ?.delete(flybitsContextPlugin)
            }
        }
    }

    /**
     *  Stop the worker [FlybitsContextPluginsWorker] assigned for the [FlybitsContextPlugin].
     *  [WorkManager] stops the worker by unique name.
     *
     *  @param workerName The worker [FlybitsContextPluginsWorker] name to be cancelled.
     */
    private fun cancelWorkByName(workerName: String) {
        WorkManager.getInstance().cancelUniqueWork(workerName)
        isRunning = false
    }

    private fun setBundle(): Bundle {
        val bundle = if (getExtras() == Bundle.EMPTY) Bundle() else getExtras()
        bundle.putLong(EXTRA_MINIMUM_REFRESH_TIME, refreshTime)
        return bundle
    }

    override fun hashCode(): Int {
        var result = refreshTimeFlex.hashCode()
        result = 31 * result + refreshTime.hashCode()
        result = 31 * result + isRunning.hashCode()
        result = 31 * result + (foregroundServiceNotification?.hashCode() ?: 0)
        result = 31 * result + (extras?.hashCode() ?: 0)
        result = 31 * result + contextPluginRetriever.hashCode()
        return result
    }

    /**
     * Builder Pattern for [FlybitsContextPlugin] to assign either
     * worker [FlybitsContextPluginsWorker] or service [FlybitsContextPluginService] to
     * this [FlybitsContextPlugin]
     */
    class Builder {
        internal var timeInSeconds: Long = 60
        internal var foregroundServiceNotification: Notification? = null
        internal var timeInSecondsFlex: Long = 60
        internal var extras: Bundle? = null

        internal val GRETZKYS_UNIQUE_ID = -99
        internal var taskWorker: Class<out FlybitsContextPluginsWorker>? = null
        internal var taskService: Class<out FlybitsContextPluginService>? = null
        internal var isWorker: Boolean = false


        /**
         * Constructor to Assign a specific Service for a plugin
         *
         * @param serviceClass The [FlybitsContextPluginService] which starts the respective Plugin
         */
        @Deprecated(
            "Using this constructor to specify Service for a Plugin is Deprecated, " +
                    "Instead provide the Class which extends a Worker [FlybitsContextPluginsWorker] as the parameter.",
            ReplaceWith("[FlybitsContextPlugin.Builder(FlybitsContextPluginsWorker)]")
        )
        constructor(serviceClass: Class<out FlybitsContextPluginService>) {
            setService(serviceClass)
        }


        /**
         * Constructor to Assign a specific Worker [FlybitsContextPluginsWorker] for a plugin
         *
         * @param workerClass The [FlybitsContextPluginsWorker] which starts the respective Plugin
         * @param isWorker Indicates weather plugin is a Worker [FlybitsContextPluginsWorker], by default,just to differentiate from service constructor.
         */
        constructor(workerClass: Class<out FlybitsContextPluginsWorker>, isWorker: Boolean = true) {
            if (isWorker) {
                setWorker(workerClass)
            }
        }


        /**
         * Constructor to Assign a specific Worker [FlybitsContextPluginsWorker] for a Reserved plugin
         *
         * @param plugin The [ReservedContextPlugin] to be started
         */
        constructor(plugin: ReservedContextPlugin) {
            assignWorker(plugin)
        }


        /**
         * Constructor to Assign a specific Worker [FlybitsContextPluginsWorker] or Service [FlybitsContextPluginService] for a Reserved plugin.
         *
         * @param plugin The [ReservedContextPlugin] to be started
         * @param isWorker The boolean if true a Worker is assigned else Service is assigned.
         */
        @Deprecated(
            "Using this constructor to assign Service for a Plugin is Deprecated, " +
                    "Instead provide the Class which extends a Worker [FlybitsContextPluginsWorker] as the parameter.",
            ReplaceWith("[FlybitsContextPlugin.Builder(ReservedContextPlugin.PLUGIN)]")
        )
        constructor(plugin: ReservedContextPlugin, isWorker: Boolean) {
            if (isWorker) {
                assignWorker(plugin)
            } else if (!isWorker) {
                assignService(plugin)
            }
        }


        /**
         * Assign the worker [FlybitsContextPluginsWorker] to this Plugin [FlybitsContextPlugin]
         *
         * @param plugin The plugin to be activated
         */
        private fun assignWorker(plugin: ReservedContextPlugin) {
            when (plugin) {
                ReservedContextPlugin.BATTERY -> setWorker(BatteryContextPluginWorker::class.java)
                ReservedContextPlugin.LOCATION -> setWorker(LocationContextPluginWorker::class.java)
                ReservedContextPlugin.CARRIER -> setWorker(CarrierContextPluginWorker::class.java)
                ReservedContextPlugin.WEATHER -> setWorker(WeatherContextPluginWorker::class.java)
                ReservedContextPlugin.NETWORK_CONNECTIVITY -> setWorker(
                    NetworkContextPluginWorker::class.java
                )

                ReservedContextPlugin.ACTIVITY -> throw InvalidContextPluginException("Worker for this Plugin Not Available.")
                ReservedContextPlugin.LANGUAGE -> setWorker(LanguageContextPluginWorker::class.java)
                ReservedContextPlugin.FITNESS -> setWorker(FitnessContextPluginWorker::class.java)
                ReservedContextPlugin.BEACON -> {
                    throw InvalidContextPluginException("Worker for this Plugin Not Available.")
                }
                else -> Logger.setTag("ReservedContextPlugin").d("Worker for this Plugin Not Available")
            }
        }


        /**
         * Assign the service to respective Plugin
         *
         * @param plugin The plugin to be activated
         */
        private fun assignService(plugin: ReservedContextPlugin) {
            when (plugin) {
                ReservedContextPlugin.BATTERY -> setService(BatteryContextPluginService::class.java)
                ReservedContextPlugin.CARRIER -> setService(CarrierContextPluginService::class.java)
                ReservedContextPlugin.LOCATION -> setService(LocationContextPluginService::class.java)
                ReservedContextPlugin.WEATHER -> setService(WeatherContextPluginService::class.java)
                ReservedContextPlugin.NETWORK_CONNECTIVITY -> setService(
                    NetworkContextPluginService::class.java
                )
                ReservedContextPlugin.ACTIVITY -> setService(ActivityContextPluginService::class.java)
                ReservedContextPlugin.LANGUAGE -> setService(LanguageContextPluginService::class.java)
                ReservedContextPlugin.FITNESS -> setService(FitnessContextPluginService::class.java)
                ReservedContextPlugin.BEACON -> {
                    setService(BeaconScanningService::class.java)
                    timeInSeconds = GRETZKYS_UNIQUE_ID.toLong()
                }
                else -> Logger.setTag("ReservedContextPlugin").d("Invalid Plugin")
            }
        }


        /**
         * Used to create the [FlybitsContextPlugin] based on the attributes set within the
         * [Builder].
         *
         * @return The [FlybitsContextPlugin] object that can registered through the
         * [com.flybits.context.ContextManager.start].
         * @throws InvalidContextPluginException If the setService is not set.
         */
        @Throws(InvalidContextPluginException::class)
        fun build(): FlybitsContextPlugin {
            if (timeInSeconds == 0L) {
                timeInSeconds = 60
            }
            if (timeInSecondsFlex == 0L) {
                timeInSecondsFlex = 60
            }

            if (isWorker) {
                timeInSeconds = 900
                timeInSecondsFlex = 300
                Logger.setTag("FlybitsContextPlugin").d("worker build")
            } else {
                Logger.setTag("FlybitsContextPlugin").d("service build")
            }
            return FlybitsContextPlugin(this)
        }

        /**
         * Set additional bundle values that can be processed by the Background/Foreground services.
         * An example of this use case is the Location Context Plugin which allows the application
         * to define the minimum displacement for location updates. This displacement can be added
         * to the [extras] Bundle.
         *
         * @param extras The additional Bundle values that should be passed to the
         * Background/Foreground service.
         * @return The [Builder] class which can be reused to add additional attributes to.
         */
        fun setExtras(extras: Bundle): Builder {
            this.extras = extras
            return this
        }


        /**
         * The refresh time in seconds which this [FlybitsContextPlugin] should wait before
         * refreshing its values.
         *
         * For Worker [FlybitsContextPluginsWorker] :
         * Please Note : Refresh time for [FlybitsContextPlugin] running with assigned [FlybitsContextPluginsWorker]
         * is minimum 900 seconds and flex time is minimum 500 seconds
         *
         * For Service [FlybitsContextPluginService] :
         * If this time is less than 60 seconds then the service will be ran in the foreground.
         * If this is the case the Notification to represent this service must be set using the
         * Builder.setNotification method.
         *
         * @param time     The refresh rate in seconds.
         * @param flexTime The flexible refresh rate in seconds that this
         * [FlybitsContextPlugin] will refresh its data.
         * @param unit     The unit of measure that time should be calculated in.
         * @return The [Builder] class which can be reused to add additional attributes to.
         */
        fun setRefreshTime(time: Long, flexTime: Long, unit: TimeUnit): Builder {
            if (timeInSeconds != GRETZKYS_UNIQUE_ID.toLong()) {
                if (isWorker) {
                    setRefreshTimeForWorker(time, unit)
                    setFlexTimeForWorker(flexTime, unit)
                } else {
                    // set refresh and flex time for Service
                    this.timeInSeconds = unit.toSeconds(time)
                    this.timeInSecondsFlex = unit.toSeconds(flexTime)
                }
            }
            return this
        }

        /**
         *  Set Refresh Time for Worker depending on TimeUnit.
         *  Please Note : Refresh time for [FlybitsContextPlugin] running with assigned [FlybitsContextPluginsWorker]
         *  is minimum 900 seconds (depending on [TimeUnit]).
         *
         *  @param time The refresh time set for worker to run.
         *  @param unit The [TimeUnit] set for worker.
         */
        private fun setRefreshTimeForWorker(time: Long, unit: TimeUnit) {
            time.let {
                when (unit) {
                    SECONDS -> if (it < 900) this.timeInSeconds =
                        unit.toSeconds(900)
                    else this.timeInSeconds = unit.toSeconds(it)
                    MINUTES -> if (it < 15) this.timeInSeconds =
                        unit.toSeconds(15)
                    else this.timeInSeconds = unit.toSeconds(it)
                    MILLISECONDS -> if (it < 900000) this.timeInSeconds =
                        unit.toSeconds(900000)
                    else this.timeInSeconds = unit.toSeconds(it)
                    else -> this.timeInSeconds = unit.toSeconds(time)
                }
            }
        }

        /**
         *  Set Flex Refresh Time for Worker depending on TimeUnit.
         *  Please Note : Flex Refresh time for [FlybitsContextPlugin] running with assigned [FlybitsContextPluginsWorker]
         *  is minimum 300 seconds (depending on [TimeUnit]).
         *
         *  @param time The flex time set for worker to run.
         *  @param unit The [TimeUnit] set for worker.
         */
        private fun setFlexTimeForWorker(flexTime: Long, unit: TimeUnit) {
            flexTime.let {
                when (unit) {
                    SECONDS -> if (it < 300) this.timeInSecondsFlex =
                        unit.toSeconds(300)
                    else this.timeInSecondsFlex = unit.toSeconds(it)
                    MINUTES -> if (it < 5) this.timeInSecondsFlex =
                        unit.toSeconds(5)
                    else this.timeInSecondsFlex = unit.toSeconds(it)
                    MILLISECONDS -> if (it < 300000) this.timeInSecondsFlex =
                        unit.toSeconds(300000)
                    else this.timeInSecondsFlex = unit.toSeconds(it)
                    else -> this.timeInSecondsFlex = unit.toSeconds(flexTime)
                }
            }
        }

        /**
         * Set the Notification to be displayed while the foreground service associated with this
         * context plugin is running. Failure to provide notification when the refresh time is less
         * than 60 seconds will result in IllegalStateException. Will be unused if refresh time >= 60.
         *
         * @param foregroundServiceNotification Notification that will be displayed for this
         * context plugin's foreground service.
         * @return The [Builder] class which can be reused to add additional attributes to.
         */
        fun setForegroundServiceNotification(foregroundServiceNotification: Notification): Builder {
            this.foregroundServiceNotification = foregroundServiceNotification
            return this
        }


        /**
         * Set the worker [FlybitsContextPluginsWorker] to specific plugin [FlybitsContextPlugin]
         *
         * @param worker The [FlybitsContextPluginsWorker] to be assigned.
         */
        private fun setWorker(worker: Class<out FlybitsContextPluginsWorker>) {
            isWorker = true
            taskWorker = worker
        }

        /**
         * Set the service [FlybitsContextPluginService] to specific plugin [FlybitsContextPlugin]
         *
         * @param service The [FlybitsContextPluginService] to be assigned.
         */
        private fun setService(service: Class<out FlybitsContextPluginService>) {
            isWorker = false
            taskService = service
        }
    }

    companion object {

        const val TABLE = "flybits_context_plugin"
        const val COLUMN_REFRESH_TIME_FLEX = "refresh_time_flex"
        const val COLUMN_REFRESH_TIME = "refresh_time"
        const val COLUMN_SERVICE = "service"
        const val COLUMN_IS_RUNNING = "running"

        const val EXTRA_NOTIFICATION = "flybits_extra_notification"
        const val EXTRA_MINIMUM_REFRESH_TIME = "flybits_minimum_refresh_time"

        const val TAG_WORK = "com.flybits.context.plugins.Work"

        private val _TAG = "FlybitsContextPlugin"

        fun getPrefName(classToBeSaved: Class<*>): String {
            return FlybitsContextPluginsWorker.PREF_CP_REFRESH_RATE_START + classToBeSaved.simpleName
        }

        /**
         * Check if the contextPluginRetriever is Worker [FlybitsContextPluginsWorker]
         *
         * @param contextPluginRetriever The class to check for its superclass.
         */
        private fun isWorkerClass(contextPluginRetriever: Class<*>): Boolean {
            return checkClass(FlybitsContextPluginsWorker::class.java, contextPluginRetriever)
        }


        /**
         * Check if the contextPluginRetriever is Service [FlybitsContextPluginService]
         *
         * @param contextPluginRetriever The class to check for its superclass.
         */
        private fun isServiceClass(contextPluginRetriever: Class<*>): Boolean {
            return checkClass(FlybitsContextPluginService::class.java, contextPluginRetriever)
        }


        /**
         * Check for superclass of class provided till it matches either Service [FlybitsContextPluginService] and Worker[FlybitsContextPluginsWorker]
         *
         * @param classLookup The class can be either Service [FlybitsContextPluginService] and Worker[FlybitsContextPluginsWorker].
         * @param contextPluginRetriever The class to check for its superclass.
         */
        internal fun checkClass(
            classLookup: Class<*>,
            contextPluginRetriever: Class<*>
        ): Boolean {
            if (contextPluginRetriever.superclass == null) {
                return false
            }
            return if (contextPluginRetriever.superclass == classLookup) true
            else checkClass(classLookup, contextPluginRetriever.superclass as Class<*>)
        }
    }
}