package com.flybits.context.models

import android.arch.persistence.room.ColumnInfo
import android.arch.persistence.room.Entity
import android.arch.persistence.room.Ignore
import android.arch.persistence.room.PrimaryKey
import android.content.Context
import android.os.Handler
import android.os.Looper
import android.os.Parcel
import android.os.Parcelable
import com.flybits.commons.library.api.FlyAway
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.http.RequestStatus
import com.flybits.commons.library.models.internal.Result
import com.flybits.context.deserializations.DeserializeEvaluatedRules
import com.flybits.context.models.results.RulesEvaluatedResult
import com.flybits.context.utils.RulesQueryParameters
import java.util.concurrent.Executors
import kotlinx.coroutines.*

/**
 * This class is used to represent an evaluated Context [Rule] within Flybits. A rule can be
 * either evaluated as true or false depending on values obtained from the Context Plugins which can
 * be obtained from a local sensor or an external one. A [Rule] needs to contain at least one
 * condition which needs to be satisfied to be true. It is important to note that a
 * [RuleEvalutated] is a simplified version of the [com.flybits.context.models.Rule]
 * object with an evaluation.
 *
 *
 * For example, an application developer may create a Context [RuleEvalutated] "Tired" that
 * is comprised of 2 context plugins; Step Counter and Sleep Quality. Therefore, an application
 * developer can create the "Tired" rule when a user's step counter is &gt; 10000 and
 * sleep quality is &lt; 6hrs.
 *
 * @param id The unique identifier that is used to identify a specific [Rule].
 * @param evaluatedTime Indicates when the [Rule] was last evaluated in Epoch time.
 * @param isLastResult Indicates whether or not the [Rule] is evaluated as true or false.
 * @param name The name of the [Rule].
 * @param dataAsString The composition of the [Rule] as a String representation.
 */
@Entity(tableName = "ruleEvaluations")
class RuleEvalutated @Ignore constructor (id: String
                                          , evaluatedTime: Long = 0
                                          , isLastResult: Boolean
                                          , name: String
                                          , dataAsString: String? = null) : Parcelable {

    @PrimaryKey
    @ColumnInfo(name = "ruleEvaluationID")
    var id: String = id

    @ColumnInfo(name = "name")
    var name: String? = name

    @ColumnInfo(name = "isLastResult")
    var isLastResult: Boolean = isLastResult

    @Ignore
    var dataAsString: String? = dataAsString
        private set

    @Ignore
    var evaluatedTime: Long = evaluatedTime
        private set

    init {
        this.name = name
        this.isLastResult = isLastResult
        this.dataAsString = dataAsString
        this.evaluatedTime = evaluatedTime
    }

    /**
     * Empty constructor for Room
     */
    constructor(): this("", "", false)

    /**
     * Constructor used for backwards compatibility with old Java version.
     */
    constructor(id: String, name: String, lastResult: Boolean): this(id, 0, lastResult, name, null)

    /**
     * Constructor used for un-flattening a [RuleEvalutated] parcel.
     *
     * @param in the parcel that contains the un-flattened [RuleEvalutated] parcel.
     */
    @Ignore
    constructor(parcel: Parcel): this(
            id = parcel.readString() ?: ""
            , evaluatedTime = parcel.readLong()
            , isLastResult = parcel.readInt() == 1
            , name = parcel.readString() ?: ""
            , dataAsString = parcel.readString())

    override fun equals(o: Any?): Boolean {
        if (this === o) return true
        if (o == null || o !is RuleEvalutated) return false
        return id == o.id
    }

    override fun toString() = "[ $id, $name : $isLastResult ]"

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    override fun describeContents(): Int {
        return 0
    }

    /**
     * Flatten this `GeoPoint` into a Parcel.
     *
     * @param out The Parcel in which the `GeoPoint` object should be written.
     * @param flags Additional flags about how the DateOfBirth object should be written.
     * May be 0 or [.PARCELABLE_WRITE_RETURN_VALUE].
     */
    override fun writeToParcel(out: Parcel, flags: Int) {
        out.writeString(id)
        out.writeLong(evaluatedTime)
        out.writeInt(if (isLastResult) 1 else 0)
        out.writeString(name)
        out.writeString(dataAsString)
    }

    companion object {

        const val API_EVALUATIONS = Rule.API + "/evaluations"
        const val API_EVALUATE_ALL = Rule.API + "/evaluateRulesForMe"

        /**
         * Parcelable.Creator that instantiates [Rule] objects
         */
        @JvmField
        val CREATOR: Parcelable.Creator<RuleEvalutated> = object : Parcelable.Creator<RuleEvalutated> {
            override fun createFromParcel(`in`: Parcel): RuleEvalutated {
                return RuleEvalutated(`in`)
            }

            override fun newArray(size: Int): Array<RuleEvalutated?> {
                return arrayOfNulls(size)
            }
        }

        /**
         * A request to get a list of [Rule]`s based on the [RulesQueryParameters]s parameter which
         * defines which options should be used to filter the list of returned parameters.
         *
         * @param mContext The context of the application.
         * @param params The [RulesQueryParameters] that indicate filters that should be placed on
         * the request.
         * @param callback The callback that indicates whether or not the request was successful or
         * whether there was an error.
         *
         * @return The [RulesEvaluatedResult] that contains the network request for getting a list of
         * [Rule].
         */
        @JvmStatic
        fun get(mContext: Context, params: RulesQueryParameters,
                callback: PagedResultCallback<RuleEvalutated>): RulesEvaluatedResult {

            val handler = Handler(Looper.getMainLooper())
            val executorService = Executors.newSingleThreadExecutor()
            val result = RulesEvaluatedResult(mContext, params, callback, executorService, handler)
            executorService.execute {
                try {
                    val getRules = FlyAway.get<RuleEvalutated>(mContext, API_EVALUATIONS, params, DeserializeEvaluatedRules(), "RuleEvaluated.get")
                    result.setResult(getRules)
                } catch (e: FlybitsException) {
                    result.setResult(Result(e, "RuleEvaluated.get() failed"))
                }
            }
            return result
        }

        /**
         *
         * This method is provided for Java compatibility. Suspend function is available
         * for Kotlin coroutine use.
         *
         * @param context The application's [Context].
         * @param callback [BasicResultCallback] that will be invoked with the result or exception.
         * @param allDevices Whether the rule's should be evaluated for all of the user's devices.
         * By default set to true.
         *
         * Evaluate all context rules for the currently authenticated [User]. Callback
         * will be invoked on main thread.
         */
        fun evaluateAll(context: Context, callback: BasicResultCallback, allDevices: Boolean = true) {
            CoroutineScope(Dispatchers.Default).launch {
                val result = evaluateAll(context, allDevices)
                CoroutineScope(Dispatchers.Main).launch {
                    if (result.status == RequestStatus.COMPLETED) callback.onSuccess()
                    else callback.onException(result.exception)
                }
            }
        }

        /**
         * Evaluate all context rules for the currently authenticated [User].
         *
         * @param context The application's [Context].
         * @param allDevices Whether the rule's should be evaluated for all of the user's devices.
         * By default set to true.
         *
         * @return [Result] of the request.
         */
        suspend fun evaluateAll(context: Context, allDevices: Boolean = true): Result<Boolean> {
            return try {
                val result = FlyAway.post<Boolean>(context, API_EVALUATE_ALL
                        , "{\"allDevices\":$allDevices}", null, "RuleEvaluated.evaluateAll()", null)
                result.setResponse(result.status == RequestStatus.COMPLETED)
                result
            } catch(e: FlybitsException) {
                val result = Result<Boolean>(e, "Failed to evaluate rules.")
                result.setResponse(false)
                return result
            }
        }
    }
}