package com.flybits.context.services

import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import android.os.Bundle
import android.support.v4.content.ContextCompat
import android.util.Log
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.flybits.commons.library.exceptions.PermissionNotSetException
import com.flybits.commons.library.logging.Logger

/**
 * The [FlybitsContextPluginsWorker] class is the base class for all
 * ContextPlugin's. It can be used to create your own Context
 * Plugin using the  `FlybitsContextPlugin.Builder#Builder(Class)` method or use one of the
 * pre-built options through `FlybitsContextPlugin.Builder#Builder(ReservedContextPlugin)`.
 */
abstract class FlybitsContextPluginsWorker(context: Context, workerParameters: WorkerParameters) :
    Worker(context, workerParameters), ContextDataRetriever {

    private val _TAG = "ContextPluginService"
    override fun doWork(): Result {
        val extras = Bundle()
        for (key in inputData.keyValueMap.keys) {
            extras.putString(key, inputData.keyValueMap.values.toString())
        }
        initialize(extras)

        return if (hasRequiredPermissions(applicationContext)) {
            run {
                try {
                    val data = getData() //getData from inputdata
                    val timeInSec = if (data.time > 0) data.time else System.currentTimeMillis() / 1000
                    data.update(applicationContext, timeInSec)
                    // Returns Success means workmanager will perform after every given interval
                    Result.success()
                } catch (exception: Exception) {
                    Logger.exception("ContextData", exception)
                    Result.retry()
                }
            }
        } else {
            // Returns failure if required permission is not granted then no need to rerun the workmanager
            Result.failure()
        }
    }

    /**
     * This checks for the permission needed for a Plugin to retrieve the Data.
     *
     * @param context Context associated with the application.
     *
     * @return true if the permission/s granted.
     */
    @Throws(PermissionNotSetException::class)
    private fun hasRequiredPermissions(context: Context): Boolean {

        val listOfPermissions = getRequiredPermissions()
        listOfPermissions.forEach { permission ->
            if (Build.VERSION.SDK_INT > Build.VERSION_CODES.LOLLIPOP_MR1) {
                return ContextCompat.checkSelfPermission(context, permission) == PackageManager.PERMISSION_GRANTED
            }
            try {
                val manager = context.packageManager
                if (manager != null) {
                    val info = manager.getPackageInfo(context.packageName, PackageManager.GET_PERMISSIONS)
                    if (info.requestedPermissions != null) {
                        info.requestedPermissions.forEach {
                            return it == permission
                        }
                    }
                }
            } catch (exception: PackageManager.NameNotFoundException) {
                Logger.exception("getPackageInfo()", exception)
                return false
            }
            Log.e(
                "Flybits", "You are attempting to register to this context plugin without enabling the " +
                        permission + " permission. This is not possible. Please include this permission in your manifest. " +
                        "For more information please visit https://developer.flybits.com/android-getting-started.html#context"
            )
            return false
        }
        return true
    }
}