package com.flybits.context;

import android.content.Context;
import android.content.SharedPreferences;
import android.database.SQLException;
import android.os.Bundle;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.QueryBuilder;
import com.flybits.commons.library.models.internal.QueryParameters;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.context.db.ContextDatabase;
import com.flybits.context.deserializations.DeserializeContextPlugins;
import com.flybits.context.models.internal.Plugin;
import com.flybits.context.plugins.FlybitsContextPlugin;
import com.google.android.gms.gcm.GcmNetworkManager;
import com.google.android.gms.gcm.GcmTaskService;
import com.google.android.gms.gcm.TaskParams;

import java.util.HashMap;
import java.util.List;
import java.util.concurrent.TimeUnit;

public class ContextPluginsService extends GcmTaskService {

    private static final String _TAG                    = "ContextPluginService";
    public static final String PREF_KEY_NUM_CPS         = "ContextPlugin-size";
    public static final String PREF_KEY_REFRESH_CPS     = "ContextPlugin-refreshTime";

    @Override
    public void onInitializeTasks() {
        super.onInitializeTasks();

        SharedPreferences preferences       = ContextScope.getContextPreferences(getBaseContext());
        HashMap<String, String> listOfClass = new HashMap<>();

        int sizeOfCustomPlugins = preferences.getInt(PREF_KEY_NUM_CPS, 0);
        long timeToRefresh = preferences.getLong(PREF_KEY_REFRESH_CPS, 86400);
        for (int i = 0; i < sizeOfCustomPlugins; i++){
            if (preferences.getString("item" + i, "").length() > 0 && preferences.getString("value"+i, "").length() > 0) {
                listOfClass.put(preferences.getString("item" + i, ""), preferences.getString("value" + i, ""));
            }
        }
        ContextManager.registerForPluginUpdates(getBaseContext(), listOfClass, timeToRefresh);
    }

    @Override
    public int onRunTask(TaskParams taskParams) {
        Logger.setTag(_TAG).d("Starting Context Plugin Retrieval...");

        HashMap<String, String> listOfCustomClasses = new HashMap<>();
        if (taskParams.getExtras() != null){

            Bundle bundle = taskParams.getExtras();
            for (int i = 0; i < bundle.getInt("listOfItems", 0); i++){
                listOfCustomClasses.put(bundle.getString("item"+i), bundle.getString("class"+i));
            }
        }
        return (getPlugins(listOfCustomClasses)) ? GcmNetworkManager.RESULT_SUCCESS : GcmNetworkManager.RESULT_FAILURE;
    }

    boolean getPlugins(HashMap<String, String> listOfCustomClasses){
        try {

            DeserializeContextPlugins deserializer      = new DeserializeContextPlugins();
            QueryBuilder builder                        = new QueryBuilder().setPaging(100, 0);
            QueryParameters params                      = new QueryParameters(builder);

            final Result<PagedResponse<Plugin>> result = FlyAway.get(getBaseContext(), Plugin.API, params, deserializer, "Plugin.get");
            if (result != null && result.getResult() != null && result.getStatus() == RequestStatus.COMPLETED){

                List<Plugin> listOfOldPlugins  = ContextDatabase.getDatabase(getBaseContext()).pluginDao().getAll();
                //Go Through all plugins previously stored on the DB
                for (Plugin plugin : listOfOldPlugins){

                    int index = result.getResult().getItems().indexOf(plugin);
                    if (index == -1) {
                        //Stop all Context Plugin that are longer stored on the Server.
                        Logger.setTag(_TAG).w("Plugin: " + plugin.getId() + " is NOT present any more");
                        stopContextPlugin(getBaseContext(), listOfCustomClasses, plugin);
                    }else{
                        Logger.setTag(_TAG).w("Plugin: " + plugin.getId() + " All Good");
                    }
                }

                //Go Through all plugins retrieved through the HTTP request
                for (Plugin plugin : result.getResult().getItems()) {

                    int index = listOfOldPlugins.indexOf(plugin);
                    if (index != -1) {

                        //Plugin was previously saved
                        Plugin oldSavedPlugin = listOfOldPlugins.get(index);

                        if (plugin.isSupported() && !oldSavedPlugin.isSupported()){
                            //If the Plugin is now enable but wasn't before, start Plugin, update ContentValues
                            startContextPlugin(listOfCustomClasses, plugin, true);
                        }

                        else if (!plugin.isSupported() && oldSavedPlugin.isSupported()){
                            //If the Plugin is not enabled any more, stop Plugin, and Remove Plugin from DB
                            stopContextPlugin(getBaseContext(), listOfCustomClasses, plugin);
                        }
                        else if (plugin.isSupported()
                                && plugin.getRefreshRate() != oldSavedPlugin.getRefreshRate()){
                            //Restart Plugin, Update DB with New Value
                            startContextPlugin(listOfCustomClasses, plugin, true);
                        }

                    }else{
                        //New Plugin
                        if (plugin.isSupported()){
                            startContextPlugin(listOfCustomClasses, plugin, false);
                        }
                    }
                }

                return true;
            }
        }catch (FlybitsException | SQLException e){
            Logger.exception("ContextRulesService.onRunTask_II", e);
        }
        return false;
    }

    private void startContextPlugin(HashMap<String, String> listOfCustomClasses, Plugin plugin, boolean isUpdate){
        //Start and Save Context Plugin
        FlybitsContextPlugin.Builder pluginToBeCreated  = null;
        if (plugin.isReserved()){
            ReservedContextPlugin availablePlugin  = ReservedContextPlugin.fromKey(plugin.getId());
            if (availablePlugin != ReservedContextPlugin.NOT_SUPPORTED){
                pluginToBeCreated   = new FlybitsContextPlugin.Builder(availablePlugin);
            }else{
                Logger.setTag(_TAG).w(plugin.getId() + " is currently not supported!");
            }

        }else{
            if (listOfCustomClasses != null && listOfCustomClasses.get(plugin.getId()) != null) {
                try {
                    Class c2 = Class.forName(listOfCustomClasses.get(plugin.getId()));
                    pluginToBeCreated = new FlybitsContextPlugin.Builder(c2);
                }catch (ClassNotFoundException e){
                    Logger.setTag(_TAG).w("The class: " + listOfCustomClasses.get(plugin.getId()) + " was not found");
                }
            }else{
                Logger.setTag(_TAG).w(plugin.getId() + " has currently has not been implemented by the Application.");
            }
        }

        if (pluginToBeCreated != null){
            pluginToBeCreated.setRefreshTime(plugin.getRefreshRate(), plugin.getRefreshRate(), TimeUnit.SECONDS);

            FlybitsContextPlugin pluginBuilt = pluginToBeCreated.build();
            ContextManager.start(getBaseContext(), pluginBuilt);
            plugin.setClassName(pluginBuilt.getService().getCanonicalName());

            if (isUpdate){
                ContextDatabase.getDatabase(getBaseContext()).pluginDao().update(plugin);
            }else {
                ContextDatabase.getDatabase(getBaseContext()).pluginDao().insert(plugin);
            }
        }
    }

    public static void stopContextPlugin(Context mContext, HashMap<String, String> listOfCustomClasses, Plugin plugin){

        FlybitsContextPlugin.Builder pluginToBeCreated  = null;
        if (plugin.isReserved()){
            ReservedContextPlugin availablePlugin  = ReservedContextPlugin.fromKey(plugin.getId());
            if (availablePlugin != ReservedContextPlugin.NOT_SUPPORTED){
                pluginToBeCreated   = new FlybitsContextPlugin.Builder(availablePlugin);
            }else{
                return;
            }

        }else{
            if (listOfCustomClasses != null && listOfCustomClasses.get(plugin.getId()) != null) {
                try {
                    Class c2 = Class.forName(listOfCustomClasses.get(plugin.getId()));
                    pluginToBeCreated = new FlybitsContextPlugin.Builder(c2);
                }catch (ClassNotFoundException e){
                    Logger.setTag(_TAG).w("The class: " + listOfCustomClasses.get(plugin.getId()) + " was not found");
                    return;
                }
            }else{
                Logger.setTag(_TAG).w(plugin.getId() + " has currently has not been implemented by the Application.");
                return;
            }
        }

        ContextManager.stop(mContext, pluginToBeCreated.build());
        ContextDatabase.getDatabase(mContext).pluginDao().delete(plugin);
    }

    @Override
    public void onDestroy() {
        Logger.setTag(_TAG).d("... Destroyed Context Plugin Retrieval");
        super.onDestroy();
    }
}
