package com.flybits.context.plugins.fitness;

import android.support.annotation.NonNull;
import com.flybits.commons.library.logging.Logger;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.common.api.ResultCallback;
import com.google.android.gms.fitness.Fitness;
import com.google.android.gms.fitness.data.*;
import com.google.android.gms.fitness.request.DataReadRequest;
import com.google.android.gms.fitness.result.DataReadResult;

import java.util.Calendar;
import java.util.Date;
import java.util.concurrent.TimeUnit;

class FitnessUtils {

    static final String _TAG    = "FitnessData";

    static void getStepsToday(GoogleApiClient client, IFitnessData callback){
        Calendar calStart   = Calendar.getInstance();
        Calendar calEnd     = Calendar.getInstance();
        Date now = new Date();
        calStart.setTime(now);

        calStart.set(Calendar.HOUR_OF_DAY, 0);
        calStart.set(Calendar.MINUTE, 0);
        calStart.set(Calendar.SECOND, 1);

        getSteps(calStart.getTimeInMillis(), calEnd.getTimeInMillis(), client, callback);
    }

    static void getSteps(long startTime, long endTime, GoogleApiClient client, final IFitnessData callback){

        DataReadRequest readRequest = new DataReadRequest.Builder()
                // The data request can specify multiple data types to return, effectively
                // combining multiple data queries into one call.
                // In this example, it's very unlikely that the request is for several hundred
                // datapoints each consisting of a few steps and a timestamp.  The more likely
                // scenario is wanting to see how many steps were walked per day, for 7 days.
                .aggregate(DataType.TYPE_STEP_COUNT_DELTA, DataType.AGGREGATE_STEP_COUNT_DELTA)
                // Analogous to a "Group By" in SQL, defines how data should be aggregated.
                // bucketByTime allows for a time span, whereas bucketBySession would allow
                // bucketing by "sessions", which would need to be defined in code.
                .bucketByTime(1, TimeUnit.DAYS)
                .setTimeRange(startTime, endTime, TimeUnit.MILLISECONDS)
                .build();

        Fitness.HistoryApi.readData(client, readRequest).setResultCallback(new ResultCallback<DataReadResult>() {
            @Override
            public void onResult(@NonNull DataReadResult dataReadResult) {
                int steps = getSteps(dataReadResult);
                if (steps == -1){
                    callback.onCancel();
                }else{
                    callback.onGetSteps(steps);
                }
            }
        });
    }

    static int getSteps(DataReadResult dataReadResult){
        int addition = 0;

        if (dataReadResult.getBuckets() == null || dataReadResult.getBuckets().size() == 0){
            return -1; //cancel
        }

        for (Bucket bucket : dataReadResult.getBuckets()) {

            if (bucket.getDataSets().size() == 0){
                return -1; //cancel
            }

            for (DataSet dataSet : bucket.getDataSets()) {

                if (dataSet.getDataPoints().size() == 0){
                    return -1; //cancel
                }

                for (DataPoint point : dataSet.getDataPoints()){

                    try{
                        addition += point.getValue(Field.FIELD_STEPS).asInt();
                    }catch (Exception e){
                        Logger.exception("FitnessUtils.getSteps", e);
                    }
                }
            }
        }

        Logger.setTag(_TAG).d( "Number Of Steps: " + addition);
        return 0;
    }

}
