package com.flybits.context.plugins.activity;

import android.app.IntentService;
import android.content.Intent;

import com.flybits.commons.library.logging.Logger;
import com.google.android.gms.location.ActivityRecognitionResult;
import com.google.android.gms.location.DetectedActivity;

import java.util.List;

/**
 * The {@code ServiceIntent} that is initiated by a pending request obtained whenever Activity
 * Recognition process has begun. This {@code ServiceIntent} is launched once the pending request is
 * completed and the result of the action has be obtained.
 */
public class DeviceRecognitionServiceIntent extends IntentService {

    private static final String _TAG = "DeviceRecognitionServiceIntent";

    /**
     * Default Constructor
     */
    public DeviceRecognitionServiceIntent() {
        super(_TAG);
    }

    @Override
    protected void onHandleIntent(Intent intent) {
        Logger.setTag(_TAG).d("onHandleIntent() - Start");

        ActivityData data = processIntent(intent);
        if (data != null) {
            Logger.setTag(_TAG).d("...Received Data: " + data.toString());
            long timeInSec                  = (data.getTime() > 0) ? data.getTime() : (System.currentTimeMillis() / 1000);

            data.update(getBaseContext(), timeInSec, null);
        }

        Logger.setTag(_TAG).d( "onHandleIntent() - End");
    }

    ActivityData processIntent(Intent intent){
        if (ActivityRecognitionResult.hasResult(intent)) {

            ActivityRecognitionResult result = ActivityRecognitionResult.extractResult(intent);
            boolean driving = false, cycling = false, stationary = false, running = false, walking = false, unknown = false;
            List<DetectedActivity> detectedActivity = result.getProbableActivities();
            for (DetectedActivity activity : detectedActivity){

                if (activity.getConfidence() > 10){
                    switch (getActivityName(activity.getType())){
                        case IN_VEHICLE:
                            driving = true;
                            break;
                        case ON_BICYCLE:
                            cycling = true;
                            break;
                        case STILL:
                            stationary = true;
                            break;
                        case RUNNING:
                            running = true;
                            break;
                        case WALKING:
                            walking = true;
                            break;
                        default:
                            unknown = true;
                    }
                }
            }
            return new ActivityData(cycling, driving, running, stationary, walking, unknown);
        } else {
            Logger.setTag(_TAG).e( "Intent had no data returned");
        }
        return null;
    }

    ActivityData.ActivityType getActivityName(int type) {
        switch (type) {
            case DetectedActivity.IN_VEHICLE:
                return ActivityData.ActivityType.IN_VEHICLE;
            case DetectedActivity.ON_BICYCLE:
                return ActivityData.ActivityType.ON_BICYCLE;
            case DetectedActivity.ON_FOOT:
                return ActivityData.ActivityType.WALKING;
            case DetectedActivity.WALKING:
                return ActivityData.ActivityType.WALKING;
            case DetectedActivity.STILL:
                return ActivityData.ActivityType.STILL;
            case DetectedActivity.TILTING:
                return ActivityData.ActivityType.STILL;
            case DetectedActivity.RUNNING:
                return ActivityData.ActivityType.RUNNING;
            case DetectedActivity.UNKNOWN:
                return ActivityData.ActivityType.UNKNOWN;
            default:
                return ActivityData.ActivityType.UNKNOWN;
        }
    }
}
