package com.flybits.context.models;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * The {@code RulePredicateQuery} class is used to create a String representation of a {@link Rule}
 * based on the {@link RulePredicate} object that constructed. These {@link RulePredicate} can be
 * combined using Ands and Ors in order to build more complex {@link Rule} queries in order to
 * suite more use cases. Additionally you can add other {@code RulePredicateQuery} objects to add
 * more complex scenarios into your application.
 */
public class RulePredicateQuery {

    private String stringRepresentation;
    private HashMap<String, String> mapOfHash;

    private RulePredicateQuery(Builder builder) {
        this.stringRepresentation   = builder.predicateAsString;
        this.mapOfHash              = builder.mapOfHash;
    }

    public HashMap<String, String> getMapOfRulePredicate(){
        return mapOfHash;
    }

    /**
     * Returns the String representation of a {@link Rule} query. This query is added to a
     * {@link Rule} in order to construct a valid {@link Rule}.
     *
     * @return The String representation of the {@link Rule} query, which indicates whether or not
     * the {@link Rule} is true.
     */
    public String getQuery() {
        return "(" + stringRepresentation + ")";
    }

    /**
     * The {@code Builder} class that defines all the possible filter options that can be used when
     * creating a {@link Rule} query.
     */
    public static class Builder {

        private String predicateAsString;
        private HashMap<String, String> mapOfHash;

        /**
         * Default constructor used to establish the minimum needs for a valid {@link Rule} query.
         *
         * @param predicate The {@link RulePredicate} used to define the base {@link Rule} query.
         */
        public Builder(RulePredicate predicate){
            predicateAsString = predicate.getString();
            mapOfHash           = new HashMap<>();
            addPredicateAsData(predicate.getPluginData());
        }

        /**
         * Build the {@link RulePredicateQuery} object to be used for constructing a {@link Rule}'s
         * composition which is made up of predicates.
         *
         * @return {@link RulePredicateQuery} object which contains all the information needed to
         * successfully construct a rule of predicates.
         */
        public RulePredicateQuery build(){
            return new RulePredicateQuery(this);
        }

        /**
         * Used to join the current query with another {@link RulePredicate}s in order to indicate
         * that both {@link RulePredicate}s should be true in order for the rule to be evaluated as
         * true.
         *
         * @param predicate The {@link RulePredicate} that should be added to the {@link Rule}
         *                  query.
         * @return A {@code Builder} object to which additional {@link RulePredicate}s can be added
         * to.
         */
        public Builder and(RulePredicate predicate){
            predicateAsString += " And " + predicate.getString();
            addPredicateAsData(predicate.getPluginData());
            return this;
        }

        /**
         * Used to join the current query with another {@link RulePredicateQuery}s in order to
         * indicate that both {@link RulePredicateQuery}s should be true in order for the rule to be
         * evaluated as true.
         *
         * @param query The {@link RulePredicateQuery} that should be added to the {@link Rule}
         *              query.
         * @return A {@code Builder} object to which additional {@link RulePredicate}s can be added
         * to.
         */
        public Builder and(RulePredicateQuery query){
            predicateAsString += " And " + query.getQuery();
            addPredicateAsData(query.getMapOfRulePredicate());
            return this;
        }

        /**
         * Used to join the current query with another {@link RulePredicateQuery}s in order to
         * indicate that either of the two {@link RulePredicateQuery}s should be true in order for
         * the rule to be evaluated as true.
         *
         * @param predicate The {@link RulePredicate} that should be added to the {@link Rule}
         *                  query.
         * @return A {@code Builder} object to which additional {@link RulePredicate}s can be added
         * to.
         */
        public Builder or(RulePredicate predicate){
            predicateAsString += " Or " + predicate.getString();
            addPredicateAsData(predicate.getPluginData());
            return this;
        }

        /**
         * Used to join the current query with another {@link RulePredicateQuery}s in order to
         * indicate that either of the two {@link RulePredicateQuery}s should be true in order for
         * the rule to be evaluated as true.
         *
         * @param query The {@link RulePredicateQuery} that should be added to the {@link Rule}
         *                  query.
         * @return A {@code Builder} object to which additional {@link RulePredicate}s can be added
         * to.
         */
        public Builder or(RulePredicateQuery query){
            predicateAsString += " Or " + query.getQuery();
            addPredicateAsData(query.getMapOfRulePredicate());
            return this;
        }

        private void addPredicateAsData(HashMap<String, String> mapOfPredicateData){
            if (mapOfPredicateData != null && mapOfPredicateData.size() > 0){

                Iterator it = mapOfPredicateData.entrySet().iterator();
                while (it.hasNext()) {
                    Map.Entry<String, String> pair = (Map.Entry<String, String>)it.next();
                    mapOfHash.put(pair.getKey(), pair.getValue());
                    it.remove();
                }

            }
        }
    }
}
