package com.flybits.context.models;

import android.arch.persistence.room.ColumnInfo;
import android.arch.persistence.room.Entity;
import android.arch.persistence.room.Ignore;
import android.arch.persistence.room.PrimaryKey;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import org.json.JSONArray;
import org.json.JSONException;

/**
 * This class represents the data that is obtained from local and external context sensors. For
 * example, if a weather context is obtaining weather information data can include information about
 * temperature, wind, precipitation, etc. Additionally, the time the sensor information is collected
 * is also stored.
 */
@Entity(tableName = "contextData")
public class BasicData implements Parcelable{

    @PrimaryKey
    @ColumnInfo(name = "contextID")
    @NonNull
    private String dataTypeID;

    @ColumnInfo(name = "isSent")
    private boolean isSent;

    @ColumnInfo(name = "lastUpdated")
    private long timestamp;

    @ColumnInfo(name = "data")
    private String valueAsString;

    public BasicData(){}

    /**
     * Constructor that initializes all variables based on a pre-defined {@code BasicData} object.
     *
     * @param object An already defined {@code BasicData} object.
     */
    @Ignore
    public BasicData(BasicData object){
        this.dataTypeID     = object.dataTypeID;
        this.timestamp      = object.timestamp;
        this.isSent         = object.isSent;
        this.valueAsString  = object.valueAsString;
    }

    /**
     * Constructor that initializes all variables.
     *
     * @param dataTypeID The unique string identifier of the
     *                   {@link com.flybits.context.plugins.FlybitsContextPlugin} being uploaded to the
     *                   server.
     * @param timestamp The epoch time of when the context data was recorded.
     * @param isSent    True is this context has already been sent to the server, false otherwise.
     * @param valueAsString The String representation of the
     *                      {@link com.flybits.context.plugins.FlybitsContextPlugin} value.
     */
    @Ignore
    public BasicData(String dataTypeID, long timestamp, boolean isSent, String valueAsString){
        this.dataTypeID     = dataTypeID;
        this.timestamp      = timestamp;
        this.isSent         = isSent;
        this.valueAsString  = valueAsString;
    }

    /**
     * Constructor used for un-flattening a {@code BasicData} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code BasicData} parcel.
     */
    @Ignore
    public BasicData(Parcel in){
        timestamp           = in.readLong();
        dataTypeID          = in.readString();
        isSent              = in.readInt() == 1;
        valueAsString       = in.readString();
    }

    /**
     * The type of data that indicates which Context Plugin this {@code BasicData} was obtained
     * from.
     *
     * @return The unique identifier of the Context Plugin
     */
    public String getDataTypeID() {
        return dataTypeID;
    }

    /**
     * The epoch representation of time that indicates when the {@code BasicData} was last updated.
     * TODO: @return
     */
    public long getTimestamp() {
        return timestamp;
    }

    /**
     * The value of the {@code BasicData} obtained from the Context plugin as its String
     * representation.
     *
     * @return The value of the Context Plugin.
     */
    public String getValueAsString() {
        return valueAsString;
    }

    /**
     * Indicates whether or not the data has been synced with the server.
     *
     * @return true if the {@code BasicData} has been synced with the server, otherwise false.
     */
    public boolean isSent() {
        return isSent;
    }

    /**
     * Sets the unique identifier of the Context Plugin.
     *
     * @param dataTypeID The unique Context identifier to represents a specific Context Plugin.
     */
    public void setDataTypeID(String dataTypeID) {
        this.dataTypeID = dataTypeID;
    }

    /**
     * Sets whether or not the {@code BasicData} has been successfully sent to the Flybits server.
     *
     * @param sent true if the {@code BasicData} was successfully sent to the server, false
     *             otherwise.
     */
    public void setSent(boolean sent) {
        isSent = sent;
    }

    /**
     * Sets the Epoch time of the date/time when the {@code BasicData} is successfully sent to the
     * server.
     *
     * @param timestamp The epoch time of the date/time.
     */
    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }

    /**
     * The value of the {@link ContextData} as a JSON String representation of the value.
     *
     * @param valueAsString The JSON string of the {@link ContextData}.
     */
    public void setValueAsString(String valueAsString) {
        this.valueAsString = valueAsString;
    }

    @Override
    public String toString() {

        //TODO: Once Arash sets up backend to only receive arrays, remove this if statement!!! - Filip
        try {
            JSONArray jsonObj  = new JSONArray(valueAsString);
            return "{" +
                    "\"timestamp\" : " + timestamp +
                    ", \"dataTypeID\" : \"" + dataTypeID + "\"" +
                    ", \"values\" : " + valueAsString +
                    "}";
        } catch (JSONException e) {}

        return "{" +
                "\"timestamp\" : " + timestamp +
                ", \"dataTypeID\" : \"" + dataTypeID + "\"" +
                ", \"value\" : " + valueAsString +
                "}";
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        BasicData basicData = (BasicData) o;

        return !(valueAsString != null ? !valueAsString.equals(basicData.valueAsString) : basicData.valueAsString != null);
    }

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    @Override
    public int describeContents() {
        return 0;
    }

    /**
     * Flatten this {@code GeoPoint} into a Parcel.
     *
     * @param out The Parcel in which the {@code GeoPoint} object should be written.
     * @param flags Additional flags about how the DateOfBirth object should be written.
     * May be 0 or {@link #PARCELABLE_WRITE_RETURN_VALUE}.
     */
    @Override
    public void writeToParcel(Parcel out, int flags) {
        out.writeLong(timestamp);
        out.writeString(dataTypeID);
        out.writeInt(isSent? 1: 0);
        out.writeString(valueAsString);
    }

    /**
     * Parcelable.Creator that instantiates {@code BasicData} objects
     */
    public static final Creator<BasicData> CREATOR = new Creator<BasicData>(){
        public BasicData createFromParcel(Parcel in){
            return new BasicData(in);
        }

        public BasicData[] newArray(int size){
            return new BasicData[size];
        }
    };
}
