package com.flybits.context;

import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.database.SQLException;

import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.QueryBuilder;
import com.flybits.commons.library.models.internal.QueryParameters;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.context.db.ContextDatabase;
import com.flybits.context.deserializations.DeserializeEvaluatedRules;
import com.flybits.context.models.RuleEvalutated;
import com.google.android.gms.gcm.GcmNetworkManager;
import com.google.android.gms.gcm.GcmTaskService;
import com.google.android.gms.gcm.TaskParams;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

public class ContextRulesService extends GcmTaskService {

    private static final String _TAG                    = "ContextRulesService";
    public static final String PREF_RULES_REFRESH_TIME  = "updateRate-ContextRules";

    @Override
    public void onInitializeTasks() {
        super.onInitializeTasks();

        SharedPreferences preferences   = ContextScope.getContextPreferences(getBaseContext());
        long numberOfSeconds = preferences.getLong(PREF_RULES_REFRESH_TIME, 3600);
        ContextManager.registerForRules(getBaseContext(), numberOfSeconds, numberOfSeconds, TimeUnit.SECONDS);
    }

    @Override
    public int onRunTask(TaskParams taskParams) {

        Logger.setTag(_TAG).d("Starting Context Rules Retrieval...");
        return (getRules()) ? GcmNetworkManager.RESULT_SUCCESS : GcmNetworkManager.RESULT_FAILURE;
    }

    boolean getRules(){
        try {

            QueryBuilder builder    = new QueryBuilder();
            Result<PagedResponse<RuleEvalutated>> result = FlyAway.get(getBaseContext(), RuleEvalutated.API_EVALUATIONS, new QueryParameters(builder), new DeserializeEvaluatedRules(), "ContextRulesService.getRules");
            if (result != null && result.getResult() != null && result.getStatus() == RequestStatus.COMPLETED){

                List<RuleEvalutated> listOfOldRules  = ContextDatabase.getDatabase(getBaseContext()).ruleDao().getAll();
                deleteRules(getBaseContext(), listOfOldRules, result.getResult().getItems());

                for (RuleEvalutated rule : result.getResult().getItems()) {
                    int index = listOfOldRules.indexOf(rule);
                    if (index != -1) {
                        RuleEvalutated oldRuleSaved = listOfOldRules.get(index);
                        updateOldRule(getBaseContext(), oldRuleSaved, rule);

                    } else {
                        Logger.setTag(_TAG).d("New Rule: " + rule.getId());
                        insertNewRule(getBaseContext(), rule);
                    }
                }
                return true;
            }
        }catch (FlybitsException | SQLException e){
            Logger.exception("ContextRulesService.onRunTask_II", e);
        }
        return false;
    }

    public void deleteRules(Context context, List<RuleEvalutated> listOfOldRules, ArrayList<RuleEvalutated> listOfNewRules){
        if (listOfNewRules == null || listOfOldRules == null){
            return;
        }

        for (RuleEvalutated oldRule : listOfOldRules){
            int index = listOfNewRules.indexOf(oldRule);
            if (index == -1) {
                ContextDatabase.getDatabase(context).ruleDao().delete(oldRule);
                Logger.setTag(_TAG).d("Deleted old Rule: " + oldRule.getId());
            }
        }
    }

    public void updateOldRule(Context context, RuleEvalutated rule, RuleEvalutated oldRuleSaved){
        if (oldRuleSaved.isLastResult() != rule.isLastResult()) {

            Intent broadcastRuleStatusUpdated = new Intent(FlyingContextConstants.BROADCAST_RULE_CHANGED);
            broadcastRuleStatusUpdated.putExtra(FlyingContextConstants.BROADCAST_RULE_CHANGED_PARAM_RULE, rule);
            context.sendBroadcast(broadcastRuleStatusUpdated);
            ContextDatabase.getDatabase(context).ruleDao().update(rule);

            Logger.setTag(_TAG).d("DATA UPDATED Amount: " + rule.getId());
        }
    }

    public void insertNewRule(Context context, RuleEvalutated rule) {
        try {
            ContextDatabase.getDatabase(context).ruleDao().insert(rule);
            Logger.setTag(_TAG).d("DATA INSERTED Amount: " + rule.getId());

            Intent broadcastRuleStatusUpdated = new Intent(FlyingContextConstants.BROADCAST_RULE_CHANGED);
            broadcastRuleStatusUpdated.putExtra(FlyingContextConstants.BROADCAST_RULE_CHANGED_PARAM_RULE, rule);
            context.sendBroadcast(broadcastRuleStatusUpdated);
        }catch (IllegalStateException exception){
            Logger.exception("ContextRulesService.insertNewRule", exception);
        }
    }

    @Override
    public void onDestroy() {
        Logger.setTag(_TAG).d("... Destroyed Context Rules Retrieval");
        super.onDestroy();
    }
}
