package com.flybits.context.plugins.battery;

import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.BatteryManager;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import com.flybits.context.models.ContextData;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * The {@code BatteryData} object is responsible for storing battery information about the device at
 * any given time. Currently the {@code BatteryData} object stored information about the device's
 * <ul>
 *     <li>Whether or not the device is charging</li>
 *     <li>The current battery percentage</li>
 * </ul>
 */
public class BatteryData extends ContextData implements Parcelable {

    private boolean isCharging;
    private int percentage;

    /**
     * Default constructor needed for generics instantiation
     */
    public BatteryData(){}

    /**
     * Constructor used for un-flattening a {@code BatteryData} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code BatteryData} parcel.
     */
    protected BatteryData(Parcel in){
        super();
        this.percentage = in.readInt();
        this.isCharging = in.readInt() == 1;
        setTime(in.readLong());
    }

    /**
     * Constructor used to define whether or not the device is charging and it's percentage based on
     * preset values.
     *
     * @param isCharging True if the device is charging, false otherwise.
     * @param percentage The device's battery percentage.
     */
    public BatteryData(boolean isCharging, int percentage){
        super();
        this.isCharging     = isCharging;
        this.percentage     = percentage;
    }

    /**
     * Constructor used to define whether or not the device is charging and it's percentage based on
     * system information associated to the {@link IntentFilter}.
     *
     * @param baseContext The current context of the application.
     * @param ifilter The {@link IntentFilter} from which battery information can be obtained.
     */
    public BatteryData(@NonNull Context baseContext, @NonNull IntentFilter ifilter) {
        super();
        if (baseContext != null && ifilter != null) {
            this.percentage = getBatteryLife(baseContext, ifilter);
            this.isCharging = isBatteryCharging(baseContext, ifilter);
        }
    }

    /**
     * Get the current battery percentage of the device.
     *
     * @return The device battery percentage of the device.
     */
    public int getPercentage() {
        return percentage;
    }

    /**
     * Indicates whether or not the device is charging.
     *
     * @return true if the device is currently charging, false otherwise.
     */
    public boolean isCharging() {
        return isCharging;
    }

    @Override
    public String toJson() {

        JSONObject object = new JSONObject();
        try {
            object.put("isCharging", isCharging);
            object.put("percentage", percentage);
        }catch (JSONException exception){}

        return object.toString();
    }

    @Override
    public void fromJson(String json) {
        try {
            JSONObject jsonObj  = new JSONObject(json);
            percentage          = jsonObj.getInt("percentage");
            isCharging          = jsonObj.getBoolean("isCharging");

        }catch (JSONException exception){}
    }

    @Override
    public String getPluginID() {
        return "ctx.sdk.battery";
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        BatteryData that = (BatteryData) o;

        if (percentage != that.percentage) return false;
        return isCharging == that.isCharging;
    }

    @Override
    public String toString(){
        return toJson();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeInt(percentage);
        dest.writeInt(isCharging ? 1 : 0);
        dest.writeLong(getTime());
    }

    /**
     * Parcelable.Creator that instantiates {@code BatteryData} objects
     */
    public static final Creator<BatteryData> CREATOR = new Creator<BatteryData>() {
        public BatteryData createFromParcel(Parcel in) {
            return new BatteryData(in);
        }

        public BatteryData[] newArray(int size) {
            return new BatteryData[size];
        }
    };

    int getBatteryLife(Context context, IntentFilter filter){

        if (context == null || filter == null){
            return -1;
        }

        Intent batteryStatus                = context.registerReceiver(null, filter);
        float batteryLifePercentage = -1.0f;

        if (batteryStatus != null) {
            int level = batteryStatus.getIntExtra(BatteryManager.EXTRA_LEVEL, -1);
            int scale = batteryStatus.getIntExtra(BatteryManager.EXTRA_SCALE, -1);

            batteryLifePercentage = level / (float) scale;
        }
        return ((int) (batteryLifePercentage * 100));
    }

    boolean isBatteryCharging(Context context, IntentFilter filter){

        if (context == null || filter == null){
            return false;
        }

        Intent batteryStatus                = context.registerReceiver(null, filter);
        if (batteryStatus != null) {
            int status = batteryStatus.getIntExtra(BatteryManager.EXTRA_STATUS, -1);
            return status == BatteryManager.BATTERY_STATUS_CHARGING || status == BatteryManager.BATTERY_STATUS_FULL;
        }
        return false;
    }

}
