package com.flybits.concierge.viewproviders

import android.content.Context
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.Button
import android.widget.ImageView
import android.widget.TextView
import com.bumptech.glide.Glide
import com.bumptech.glide.request.RequestOptions
import com.flybits.android.kernel.ContentAnalytics
import com.flybits.concierge.FlybitsNavigator
import com.flybits.concierge.FlybitsViewProvider
import com.flybits.concierge.R
import com.flybits.concierge.UserActionHandler
import com.flybits.concierge.activities.ActionCardDetailActivity
import com.flybits.concierge.models.ActionCard
import com.flybits.concierge.viewactionhandlers.ActionCardUserActionHandler
import com.flybits.concierge.viewholders.FlybitsViewHolder


class ActionCardViewProvider internal constructor(
    private val contentAnalytics: ContentAnalytics?
    , private val userActionHandler: UserActionHandler<ActionCard>
) : FlybitsViewProvider<ActionCard> {

    /**
     * Constructor used for instantiating the [ActionCardViewProvider] with the default [UserActionHandler] for [ActionCard].
     *
     * @param context The application's [Context].
     */
    constructor(context: Context) : this(ContentAnalytics(context), ActionCardUserActionHandler())

    /**
     * Constructor used for instantiating the [ActionCard] with a specific [ActionCardViewProvider].
     *
     * @param context The application's [Context].
     *
     * @param userActionHandler The [UserActionHandler] which will respond to user actions in the UI.
     * When overriding only a subset of the default handling logic, extend and override [ActionCardViewProvider].
     */
    constructor(context: Context, userActionHandler: UserActionHandler<ActionCard>)
            : this(ContentAnalytics(context), userActionHandler)


    override fun getClassType(): Class<ActionCard> {
        return ActionCard::class.java
    }

    override fun getContentType(): String {
        return CONTENT_TYPE
    }

    override fun onCreateViewHolder(parent: ViewGroup): FlybitsViewHolder<ActionCard> {
        val view = LayoutInflater.from(parent.context)
            .inflate(R.layout.flybits_streamline_buttons, parent, false)
        return ActionCardViewHolder(view, contentAnalytics, userActionHandler)
    }


    companion object {

        const val CONTENT_TYPE = "concierge-card-buttons"

        private fun configureView(
            data: ActionCard,
            actionCardView: View,
            userActionHandler: UserActionHandler<ActionCard>,
            flybitsNavigator: FlybitsNavigator,
            contentAnalytics: ContentAnalytics?
        ) {
            val bannerView: ImageView =
                actionCardView.findViewById(R.id.flybits_streamline_button_imgHeader)
            val headerView: TextView =
                actionCardView.findViewById(R.id.flybits_streamline_button_header)
            val titleView: TextView =
                actionCardView.findViewById(R.id.flybits_streamline_button_title)
            val descriptionView: TextView =
                actionCardView.findViewById(R.id.flybits_streamline_button_description)
            val primaryButton: Button =
                actionCardView.findViewById(R.id.flybits_streamline_button_primary_action)
            val secondaryButton: Button =
                actionCardView.findViewById(R.id.flybits_streamline_button_secondary_button)

            data.image?.value.let {
                if (it.isNullOrEmpty()) {
                    bannerView.visibility = View.GONE
                } else {
                    Glide.with(bannerView.context).load(it)
                        .apply(RequestOptions().centerCrop()).into(bannerView)
                }
            }

            data.header?.value.let {
                if (it.isNullOrEmpty()) {
                    headerView.visibility = View.GONE
                } else {
                    headerView.text = it
                }
            }

            titleView.text = data.title?.value

            data.description?.value.let {
                if (it.isNullOrEmpty()) {
                    descriptionView.visibility = View.GONE
                } else {
                    descriptionView.text = it
                }
            }

            when (data.buttons?.list?.size) {
                0 -> {
                    primaryButton.visibility = View.GONE
                    secondaryButton.visibility = View.GONE
                }
                1 -> {
                    primaryButton.let {
                        it.visibility = View.VISIBLE
                        it.text = data.buttons?.list?.get(0)?.text?.value.toString()
                    }
                    secondaryButton.visibility = View.GONE
                }
                2 -> {
                    primaryButton.let {
                        it.visibility = View.VISIBLE
                        it.text = data.buttons?.list?.get(0)?.text?.value.toString()
                    }
                    secondaryButton.let {
                        it.visibility = View.VISIBLE
                        it.text = data.buttons?.list?.get(1)?.text?.value.toString()
                    }
                }
            }

            primaryButton.setOnClickListener {
                userActionHandler.onUserAction(
                    ActionCardUserActionHandler.PRIMARY_CLICKED,
                    data,
                    flybitsNavigator
                )
            }

            secondaryButton.setOnClickListener {
                userActionHandler.onUserAction(
                    ActionCardUserActionHandler.SECONDARY_CLICKED,
                    data,
                    flybitsNavigator
                )
            }
        }
    }


    class ActionCardViewHolder(
        val view: View,
        val contentAnalytics: ContentAnalytics?,
        private val userActionHandler: UserActionHandler<ActionCard>
    ) : FlybitsViewHolder<ActionCard>(view) {
        private val single: View = view.findViewById(R.id.flybits_streamline_button_single)
        private val emptyView: View =
            view.findViewById(R.id.smartrewards_streamline_button_emptyTxt)

        override fun bindData(data: ActionCard, flybitsNavigator: FlybitsNavigator) {
            single.visibility = View.GONE
            emptyView.visibility = View.GONE

            when (data.title?.value) {
                null -> {
                    emptyView.visibility = View.VISIBLE
                    single.visibility = View.GONE
                }
                else -> {
                    emptyView.visibility = View.GONE
                    single.visibility = View.VISIBLE
                    configureView(
                        data,
                        view,
                        userActionHandler,
                        flybitsNavigator,
                        contentAnalytics
                    )
                }
            }
        }
    }

    override fun onPushOpened(model: ActionCard, flybitsNavigator: FlybitsNavigator) {
        model.let {
            val bundle = Bundle()
            bundle.putParcelable(ActionCardDetailActivity.EXTRA_ACTIONCARD_DETAILS, it)
            bundle.putParcelable(
                ActionCardDetailActivity.EXTRA_USER_ACTION_HANDLER,
                userActionHandler
            )
            flybitsNavigator.openActivity(ActionCardDetailActivity::class.java, bundle)
        }
    }
}