package com.flybits.concierge.viewmodels

import android.arch.lifecycle.LiveData
import android.arch.lifecycle.MutableLiveData
import android.arch.lifecycle.ViewModel
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.FlybitsConcierge
import com.flybits.concierge.R
import com.flybits.concierge.ResourceProvider
import com.flybits.concierge.enums.DisplayType

const val TAG_OPTIN_MODEL = "OptInModel"

class OptInViewModel(
    private val flybitsConcierge: FlybitsConcierge,
    resourceProvider: ResourceProvider
) : ViewModel() {

    data class OptInResult(val success: Boolean, val error: FlybitsException? = null)
    data class Span(
        val tncStartIndex: Int,
        val tncEndIndex: Int,
        val privacyPolicyStartIndex: Int,
        val privacyPolicyEndIndex: Int,
        val text: String
    )

    val optInTitle: String
    val footerSpan: LiveData<Span> = MutableLiveData()
    val checkboxState: LiveData<Boolean> = MutableLiveData()
    val optInText: String

    init {
        val termsAndConditionsText =
            resourceProvider.getString(R.string.flybits_con_terms_and_conditions)
        val privacyPolicyText =
            resourceProvider.getString(R.string.flybits_con_settings_privacy_policy)
        optInText = flybitsConcierge.configuration.optInTermsOfService
            ?: resourceProvider.getString(R.string.flybits_con_opt_in_body_default_text)
        optInTitle = flybitsConcierge.configuration.optInTitle
            ?: resourceProvider.getString(R.string.flybits_con_opt_in_title_default_text)
        val text = String.format(
            resourceProvider.getString(R.string.flybits_con_terms_and_conditions_body)
            , termsAndConditionsText, privacyPolicyText
        )
        val tncStartIndex =
            text.length - termsAndConditionsText.length - privacyPolicyText.length - 1
        val tncEndIndex = text.length - privacyPolicyText.length - 1
        val privacyPolicyStartIndex = text.length - privacyPolicyText.length

        (footerSpan as MutableLiveData).postValue(
            Span(tncStartIndex, tncEndIndex, privacyPolicyStartIndex, text.length, text)
        )

        (checkboxState as MutableLiveData).postValue(false)
    }

    fun checkboxSelected() {
        (checkboxState as MutableLiveData).postValue(checkboxState.value?.not())
    }

    fun optIn(): LiveData<OptInResult> {
        val liveData = MutableLiveData<OptInResult>()
        flybitsConcierge.optIn(object : BasicResultCallback {
            override fun onException(exception: FlybitsException) {
                liveData.postValue(OptInResult(false, exception))
            }

            override fun onSuccess() {
                liveData.postValue(OptInResult(true))
            }
        })
        return liveData
    }

    fun declinedOptIn() {
        if (flybitsConcierge.declineOptInCallBack != null) {
            Logger.appendTag(TAG_OPTIN_MODEL).d("Opt In declined successful.")
            flybitsConcierge.declineOptInCallBack.onSuccess()
        } else {
            Logger.appendTag(TAG_OPTIN_MODEL).d("Opt In declined callback returned null.")
        }
    }

    fun optInErrorAction() {
        flybitsConcierge.broadcastOptedInStateChangeFor2Phase(DisplayType.SHOW_OPT_IN)
    }

    fun optInInitiated() {
        flybitsConcierge.optIn2PhaseCallBack.onSuccess()
    }
}