package com.flybits.concierge.fragments

import android.arch.lifecycle.Observer
import android.os.Bundle
import android.support.design.widget.Snackbar
import android.support.v4.app.Fragment
import android.support.v4.widget.SwipeRefreshLayout
import android.support.v7.widget.LinearLayoutManager
import android.support.v7.widget.LinearSmoothScroller
import android.support.v7.widget.RecyclerView
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import com.afollestad.materialdialogs.MaterialDialog
import com.flybits.android.push.models.Push
import com.flybits.commons.library.api.OptInStateChangeListener
import com.flybits.commons.library.api.OptInStateObservable
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.*
import com.flybits.concierge.adapters.NotificationAdapter
import com.flybits.concierge.models.NotificationContent
import com.flybits.concierge.repository.RepositoryResponse
import com.flybits.concierge.viewmodels.NotificationsViewModel
import com.flybits.concierge.viewmodels.NotificationsViewModelFactory


/**
 * This fragment displays up to 20 most recent [NotificationContent] that the user has received.
 * Only notifications that are tied to content are shown, and these content must
 * have a [FlybitsViewProvider] registered for them to be shown.
 */
class NotificationsFragment: Fragment() {

    companion object {
        @JvmStatic
        fun newInstance(push: Push? = null): NotificationsFragment {
            val fragment = NotificationsFragment()
            if (push != null){
                val bundle = Bundle()
                bundle.putParcelable(ConciergeConstants.PUSH_EXTRA, push)
                fragment.arguments = bundle
            }
            return fragment
        }
    }

    private var notificationsViewModel: NotificationsViewModel? = null
    private var notificationsAdapter: NotificationAdapter? = null
    private var swipeToRefresh: SwipeRefreshLayout? = null
    private var recyclerView: RecyclerView? = null
    private val optInSubscriber: OptInStateChangeListener


    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View? {
        return inflater.inflate(R.layout.flybits_con_fragment_category, container, false)
    }

    init{
        optInSubscriber = object: OptInStateChangeListener {
            override fun onOptedInStateChange(optedIn: Boolean) {
                if (!optedIn) {
                    val manager = childFragmentManager
                    manager.beginTransaction()
                        .add(OptInFragment.newInstance(),"OptIn")
                        .addToBackStack(null)
                        .commit()
                    activity?.finish()
                }
            }
        }
    }

    override fun onResume() {
        super.onResume()
        OptInStateObservable.subscribe(optInSubscriber)
    }

    override fun onPause() {
        super.onPause()
        OptInStateObservable.unsubscribe(optInSubscriber)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        val navigator: FlybitsNavigator = when {
            activity is FlybitsNavigator -> activity as FlybitsNavigator
            parentFragment is FlybitsNavigator -> parentFragment as FlybitsNavigator
            else -> throw FlybitsException("No flybits navigator defined. Activity or hosting " +
                    "fragment must implement FlybitsNavigator interface")
        }

        notificationsViewModel = NotificationsViewModelFactory(context!!
                , FlybitsConcierge.with(context), navigator).create(NotificationsViewModel::class.java)

        swipeToRefresh = view.findViewById(R.id.flybits_con_swipe_ref_layout)
        swipeToRefresh?.setOnRefreshListener {
            loadNotifications()
        }
        swipeToRefresh?.isRefreshing = true

        recyclerView = view.findViewById(R.id.flybits_con_fragment_feed_lytRecycler)
        recyclerView?.visibility = View.VISIBLE
        recyclerView?.layoutManager = LinearLayoutManager(context!!)

        notificationsAdapter = NotificationAdapter(FlybitsConcierge.with(context), navigator)
        recyclerView?.adapter = notificationsAdapter

        val pushExtra = arguments?.get(ConciergeConstants.PUSH_EXTRA) as Push?
        loadNotifications(pushExtra)

        //Let view model handle passed push
        pushExtra?.let { push  ->
            arguments?.remove(ConciergeConstants.PUSH_EXTRA)
        notificationsViewModel?.handlePassedPush(push)?.observe(this, Observer {
                if (it?.data == false){
                    context?.let { nonNullContext ->
                        MaterialDialog.Builder(nonNullContext)
                                .title(R.string.flybits_con_error)
                                .content(getString(R.string.flybits_con_notification_content_error))
                                .positiveText(android.R.string.ok)
                                .show()
                    }

                }
            })
        }
    }

    //Don't reuse this pattern, it sucks, ViewModel should have full control, this should only be called once at the start to bind
        private fun loadNotifications(scrollTo: Push? = null){
        notificationsViewModel?.getNotificationContents(scrollTo)?.observe(this
                , Observer<RepositoryResponse<NotificationsViewModel.ViewModelNotificationContent>> {
            Logger.d("Notifications: loaded = $it")

            //Return if context null otherwise proceed saving non null context variable
            val nonNullContext = context ?: return@Observer

            it?.data?.data?.let { notificationBaseTemplateList ->
                toggleEmptyState(notificationBaseTemplateList.isEmpty())
                notificationsAdapter?.submitList(notificationBaseTemplateList)

                //Scroll such that tile appears on top of the screen unless its at the bottom of the recycler view
                it.data.positionOfPush?.let { position ->
                    val smoothScroller = object : LinearSmoothScroller(nonNullContext) {
                        override fun getVerticalSnapPreference(): Int {
                            return LinearSmoothScroller.SNAP_TO_START
                        }
                    }.apply {
                        targetPosition = position
                    }

                    //Need a delay to allow for views to draw
                    recyclerView?.postDelayed({
                        recyclerView?.layoutManager?.startSmoothScroll(smoothScroller)
                    },500)
                }
            }

            it?.error?.let {
                view?.let { nonNullView ->
                    toggleEmptyState(true)
                    Snackbar.make(nonNullView, getString(R.string.flybits_con_notifications_error), Snackbar.LENGTH_LONG).show()
                }
            }

            swipeToRefresh?.isRefreshing = false
        })
    }

    private fun toggleEmptyState(on: Boolean){
        if (on){
            view?.findViewById<View>(R.id.flybits_con_fragment_feed_lytNoData)?.visibility = View.VISIBLE
            view?.findViewById<View>(R.id.flybits_con_fragment_feed_lytRecycler)?.visibility = View.GONE
        }else{
            view?.findViewById<View>(R.id.flybits_con_fragment_feed_lytNoData)?.visibility = View.GONE
            view?.findViewById<View>(R.id.flybits_con_fragment_feed_lytRecycler)?.visibility = View.VISIBLE
        }
    }
}