package com.flybits.concierge.viewpopulators

import android.content.Intent
import android.net.Uri
import android.os.Bundle
import android.text.SpannableString
import android.text.TextUtils
import android.text.style.UnderlineSpan
import android.view.View
import android.widget.ImageView
import android.widget.TextView
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DecodeFormat
import com.bumptech.glide.request.RequestOptions
import com.flybits.concierge.FlybitsNavigator
import com.flybits.concierge.Utils
import com.flybits.concierge.activities.EventDetailsActivity
import com.flybits.concierge.models.Event

/**
 * Handles logic to bind Event data with Event views
 */
class EventsViewPopulator {

    /**
     *  Binds the event model data with Event views
     *
     * @param event event data to bind to the UI
     * @param flybitsNavigator enables capability to interact with event (open url, open activity etc)
     *
     * Other parameters are views related to each Event item
     */
    fun populate(event: Event, flybitsNavigator: FlybitsNavigator?, titleView: TextView?, locationView: TextView, dateView: TextView, imgHeader: ImageView, eventView: View?, addressView: TextView?, ctaView: TextView?, phoneView: TextView?, descriptionView: TextView?) {
        val options = RequestOptions().centerCrop().format(DecodeFormat.PREFER_RGB_565)

        Glide.with(imgHeader.context)
            .load(event.coverPhotoURL)
            .apply(options)
            .into(imgHeader)

        titleView?.text = if (!TextUtils.isEmpty(event.title.value)) event.title.value else ""
        locationView.text = if (!TextUtils.isEmpty(event.venueDescription.value)) event.venueDescription.value else ""
        dateView.text = Utils.getDateRange(event.getStartDate(), event.getEndDate())

        if (eventView != null && flybitsNavigator != null) {
            eventView.setOnClickListener {
                val extras = Bundle()
                extras.putParcelable(EventDetailsActivity.EXTRA_EVENT_DETAILS, event)
                flybitsNavigator.openActivity(EventDetailsActivity::class.java, extras)
            }
        }

        if (addressView != null) {
            if (descriptionView != null) {
                val content = SpannableString(event.address)
                content.setSpan(UnderlineSpan(), 0, content.length, 0)
                addressView.text = content
            } else {
                addressView.text = event.address
            }
        }

        if (ctaView != null) {
            ctaView.visibility = if (!TextUtils.isEmpty(event.url)) View.VISIBLE else View.INVISIBLE

            ctaView.setOnClickListener { v ->
                val url = event.url
                val intent = Intent(Intent.ACTION_VIEW)
                intent.data = Uri.parse(url)
                v.context.startActivity(intent)
            }
        }

        if (phoneView != null) {
            if (!TextUtils.isEmpty(event.phoneNumber)) {
                if (descriptionView != null) {
                    val content = SpannableString(event.phoneNumber)
                    content.setSpan(UnderlineSpan(), 0, content.length, 0)
                    phoneView.text = content
                } else {
                    phoneView.text = event.phoneNumber
                }
            } else {
                phoneView.setText(android.R.string.unknownName)
            }
        }

        descriptionView?.text = event.description.value
    }

    /**
     * Bind events' data with the views provided and adds it to the container
     *
     * @param data events' model data
     * @param flybitsNavigator enables capability to interact with event (open url, open activity etc)
     *
     * Other parameters are views related to each Event item
     */
    fun populate(data: Event, flybitsNavigator: FlybitsNavigator, titleView: TextView, locationView: TextView, dateView: TextView, imgHeader: ImageView, eventView: View) {
        populate(data, flybitsNavigator, titleView, locationView, dateView, imgHeader, eventView, null, null, null, null)
    }

}