package com.flybits.concierge.services

import android.content.Context
import androidx.work.ListenableWorker
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.flybits.android.kernel.db.KernelDatabase
import com.flybits.android.kernel.models.Content
import com.flybits.android.kernel.utilities.ContentParameters
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.models.internal.Pagination
import com.flybits.concierge.R
import com.flybits.concierge.models.CategorySettings
import com.flybits.concierge.repository.ProjectSettingsGetter
import com.flybits.concierge.repository.category.CategoryDatabase
import com.flybits.concierge.repository.category.CategorySettingsRepository
import com.flybits.concierge.repository.content.ContentGetter
import com.flybits.concierge.repository.content.ContentRepository
import com.flybits.concierge.viewmodels.CategoryViewModel
import com.flybits.internal.db.CommonsDatabase
import java.util.*

/**
 * Responsible for loading the data required by the views into the local storage's from the server
 *
 * This should ensure that categories and some of their content will appear fairly quick majority of the time.
 */
class PreloadingWorker(context: Context, workerParams: WorkerParameters) : Worker(context, workerParams) {

    companion object {
        const val MAX_CATEGORY_PRELOAD = 5
    }

    override fun doWork(): ListenableWorker.Result {
        Logger.d("${PreloadingWorker::class.java.simpleName}: Preloading worker running")
        val categoryDao = CategoryDatabase.getInstance(applicationContext)?.categoryDao()
                ?: return Result.failure()

        val projectSettingsGetter = ProjectSettingsGetter(applicationContext)
        val categorySettingsRepository = CategorySettingsRepository(categoryDao, projectSettingsGetter)

        val contentGetter = ContentGetter(applicationContext)
        val contentDao = KernelDatabase.getDatabase(applicationContext).contentDao()
        val cachingEntryDao = CommonsDatabase.getDatabase(applicationContext).cachingEntryDAO()
        val contentRepository = ContentRepository(contentGetter, contentDao, cachingEntryDao)

        categorySettingsRepository.getCategorySettingsRemote(callback = object: ObjectResultCallback<CategorySettings>{
            override fun onSuccess(item: CategorySettings) {
                Logger.d("${PreloadingWorker::class.java.simpleName}: Successfully preloaded categories: ${item.categories}")

                //Load the home tab
                loadContentPage(contentRepository = contentRepository)

                item.let { categorySettingsNotNull ->
                    val sublistSize = if (MAX_CATEGORY_PRELOAD > categorySettingsNotNull.categories.size) categorySettingsNotNull.categories.size else MAX_CATEGORY_PRELOAD
                    //sublisting so that you don't overburden the app loading a lot of categories
                    categorySettingsNotNull.categories.subList(0, sublistSize).forEach {
                        loadContentPage(it.name, contentRepository)
                    }
                }
            }

            override fun onException(exception: FlybitsException) {
                Logger.d("${PreloadingWorker::class.java.simpleName}: Error preloading categories")
            }

        })
        return Result.success()
    }

    private fun loadContentPage(label: String? = null, contentRepository: ContentRepository){
        val paramsBuilder = ContentParameters.Builder()
                .setPaging(CategoryViewModel.PAGE_SIZE.toLong(), 0)
                .setCaching(applicationContext.getString(R.string.flybits_con_cache_key), CategoryViewModel.INITIAL_LOAD_SIZE)

        if (label != null){
            paramsBuilder.setLabels(label)
        }

        contentRepository.getContent(paramsBuilder.build(), object: PagedResultCallback<Content>{
            override fun onSuccess(items: ArrayList<Content>, pagination: Pagination) {
                Logger.d("${PreloadingWorker::class.java.simpleName}: Successfully preloaded content for label: $label")
            }

            override fun onLoadedAllItems() {
            }

            override fun onException(exception: FlybitsException) {
                Logger.d("${PreloadingWorker::class.java.simpleName}: Error preloading content for label: $label")
            }

        })
    }
}
