package com.flybits.concierge.fragments

import android.arch.lifecycle.Observer
import android.arch.lifecycle.ViewModelProviders
import android.os.Bundle
import android.support.design.widget.TabLayout
import android.support.v4.app.Fragment
import android.support.v4.app.FragmentManager
import android.support.v4.app.FragmentStatePagerAdapter
import android.support.v4.view.PagerAdapter
import android.support.v4.view.ViewPager
import android.util.TypedValue
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.R
import com.flybits.concierge.models.Category
import com.flybits.concierge.viewmodels.FeedHolderViewModel
import com.flybits.concierge.viewmodels.FeedHolderViewModelFactory

/**
 * FragmentPagerAdapter used for displaying the appropriate [CategoryFragment] in the
 * associated [ViewPager].
 */
class CategoryFragmentPagerAdapter(fragmentManager: FragmentManager, var categoryList: List<Category>, val showSettings: Boolean): FragmentStatePagerAdapter(fragmentManager) {

    init {
        Logger.d("${CategoryFragmentPagerAdapter::class.java.simpleName}: Creating, categoryList: $categoryList")
    }

    override fun getPageTitle(position: Int): CharSequence? {
        return if (showSettings){
            if (position == 0){
                null
            } else {
                categoryList[position-1].name //menu tab is added
            }
        } else {
            categoryList[position].name //no menu tab
        }

    }

    override fun getItemPosition(item: Any): Int {
        return PagerAdapter.POSITION_NONE
    }

    override fun getItem(position: Int): Fragment {
        return if (showSettings){
            if (position == 0){
                SettingsFragment.newInstance(true)
            } else {
                CategoryFragment.newInstance(categoryList[position-1]) //menu tab added
            }
        } else {
            CategoryFragment.newInstance(categoryList[position]) //no menu tab
        }

    }

    override fun getCount(): Int{
        return if (showSettings){
            categoryList.size + 1 //Append one since menu tab is present
        } else {
            categoryList.size
        }
    }

}

/**
 * Fragment responsible for displaying the concierge feed which includes a [TabLayout]
 * which contains categories, and a [ViewPager] which displays the categories [CategoryFragment].
 *
 * To instantiate [FeedHolderFragment] use the `newInstance` method.
 */
class FeedHolderFragment: Fragment() {

    private var firstStart = true

    companion object {

        const val ARG_SHOW_MORE_TAB = "flybits_con_arg_show_more_tab"
        const val VIEW_PAGER_OFF_SCREEN_PAGE_LIMIT = 4

        /**
         * Create instance of [FeedHolderFragment].
         *
         * @param showMoreTab Whether the tab containing settings, and notification should be
         * displayed.
         *
         * @return New instance of [FeedHolderFragment]
         */
        fun newInstance(showMoreTab: Boolean): FeedHolderFragment{
            val fragment = FeedHolderFragment()
            val bundle = Bundle()
            bundle.putBoolean(ARG_SHOW_MORE_TAB, showMoreTab)
            fragment.arguments = bundle
            return fragment
        }
    }

    private var feedHolderViewModel: FeedHolderViewModel? = null

    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View? {
        return inflater.inflate(R.layout.flybits_con_fragment_feed_holder, container, false)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        Logger.d("FeedHolderFragment: view created!")
        val viewPager: ViewPager = view.findViewById(R.id.category_view_pager)
        val tabLayout = view.findViewById<TabLayout>(R.id.category_tab_layout)
        viewPager.offscreenPageLimit = VIEW_PAGER_OFF_SCREEN_PAGE_LIMIT

        feedHolderViewModel = ViewModelProviders.of(this, FeedHolderViewModelFactory(context!!)).get(FeedHolderViewModel::class.java)
        feedHolderViewModel?.syncCategories()
        feedHolderViewModel?.getCategories()?.observe(this, Observer { categoryList ->
            Logger.d("FeedHolderFragment: Got categories: $categoryList")
            //add the default tab to the front
            categoryList?.add(0, Category(null, getString(R.string.flybits_con_default_tab_name), emptySet()))
            childFragmentManager.let {
                if (viewPager.adapter == null || (viewPager.adapter as CategoryFragmentPagerAdapter).categoryList != categoryList){
                    val moreTabEnabled = arguments?.getBoolean(ARG_SHOW_MORE_TAB, false) == true

                    val pagerAdapter = CategoryFragmentPagerAdapter(it, categoryList ?: emptyList(), moreTabEnabled)
                    viewPager.adapter = pagerAdapter
                    tabLayout.setupWithViewPager(viewPager)

                    if (moreTabEnabled){
                        //Set menu tab custom view
                        val menuTab = tabLayout.getTabAt(0)
                        val menuTabLayout = LayoutInflater.from(context).inflate(R.layout.flybits_con_menu_tab, tabLayout, false)
                        menuTab?.customView = menuTabLayout
                        val moreTabView = (tabLayout.getChildAt(0) as ViewGroup).getChildAt(0)
                        moreTabView.setPadding(0, moreTabView.paddingTop, 0, moreTabView.paddingBottom)
                        val px = TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, 50f, resources.displayMetrics
                        )
                        moreTabView.layoutParams.width = px.toInt()

                        //Select home tab
                        tabLayout.getTabAt(1)?.select()
                    }
                }
            }
        })
    }

    override fun onStart() {
        super.onStart()

        //Refresh categories if activity is brought to start state ONLY after it is called more than once
        //This refreshes the content when the app is brought into the foreground from the background
        if (firstStart){
            firstStart = false
        }else{
            feedHolderViewModel?.syncCategories()
        }
    }
    
}