package com.flybits.concierge;

import android.content.Context;
import android.content.res.Resources;
import android.content.res.XmlResourceParser;

import com.flybits.commons.library.logging.Logger;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedHashMap;

public class ConciergeConfiguration {

    public static final String KEY_TWITTER_CONSUMER = "key_twitter_consumer";
    public static final String KEY_TWITTER_SECRET = "key_twitter_secret";
    public static final String KEY_TIME_TO_UPLOAD_CONTEXT = "timeToUploadContext";
    public static final int DEFAULT_TIME_TO_UPLOAD_CONTEXT = 10;

    private String projectID;
    private String tncUrl = null;
    private String privacyPolicyUrl = null;
    private boolean isAnonymous = true;
    private HashMap<String, String> apiMap = new LinkedHashMap<>();
    private long timeToUploadContext;

    private ConciergeConfiguration()
    {
    }

    private ConciergeConfiguration(Builder builder)
    {
        this.projectID = builder.projectID;
        this.isAnonymous = builder.isAnonymous;
        this.tncUrl = builder.tncUrl;
        this.apiMap = builder.apiMap;
        this.timeToUploadContext = builder.timeToUploadContext;
    }

    static ConciergeConfiguration createFromXML(Context context, int cfgResource) {
        Resources res = context.getResources();
        XmlResourceParser xrp = res.getXml(cfgResource);

        String projectID;
        boolean isAnon;
        boolean tncRequired;

        boolean scanninConcierge = false;
        boolean doneScan = false;

        ConciergeConfiguration cfg = new ConciergeConfiguration();

        try {
            int eventType = xrp.getEventType();
            while (eventType != XmlPullParser.END_DOCUMENT) {
                // instead of the following if/else if lines
                // you should custom parse your xml
                if (eventType == XmlPullParser.START_DOCUMENT) {
                    System.out.println("Start document");
                } else if (eventType == XmlPullParser.START_TAG) {
                    switch (xrp.getName())
                    {
                        case "concierge": {
                            scanninConcierge = true;
                            cfg.projectID = xrp.getAttributeValue(null, "projectID");
                            cfg.isAnonymous = xrp.getAttributeBooleanValue(null, "anonymous", false);
                            cfg.tncUrl = xrp.getAttributeValue(null, "tncUrl");
                            cfg.privacyPolicyUrl = xrp.getAttributeValue(null,"privacyPolicyUrl");
                            cfg.timeToUploadContext = xrp.getAttributeIntValue(null, KEY_TIME_TO_UPLOAD_CONTEXT, DEFAULT_TIME_TO_UPLOAD_CONTEXT);
                            break;
                        }
                        case "api": {
                            if (scanninConcierge)
                            {
                                String apiName = xrp.getAttributeValue(null, "name");
                                String apiKey = xrp.getAttributeValue(null, "value");
                                cfg.apiMap.put(apiName, apiKey);
                            }
                            break;
                        }
                    }
                } else if (eventType == XmlPullParser.END_TAG) {
                    String endTag = xrp.getName();
                    switch(endTag)
                    {
                        case "concierge":
                            doneScan = true;
                            break;
                    }
                } else if (eventType == XmlPullParser.TEXT) {
                    System.out.println("Text " + xrp.getText());
                }
                if (doneScan)
                    break;
                eventType = xrp.next();
            }
        } catch (XmlPullParserException e) {
            throw new IllegalStateException("Concierge: Cfg syntax error.");
        } catch (IOException e) {
            Logger.exception(ConciergeConfiguration.class.getSimpleName(), e);
        }
        xrp.close();

        //Check if Project ID is null
        if (cfg.projectID == null)
            throw new IllegalStateException("Developer: The projectID attribute didn't exist.");

        return cfg;
    }

    public String getProjectID()
    {
        return projectID;
    }

    public String getTNCUrl() {
        return tncUrl;
    }

    public String getPrivacyPolicyUrl(){
        return privacyPolicyUrl;
    }

    public long getTimeToUploadContext(){
        return timeToUploadContext;
    }

    public boolean isAnonymous() {
        return isAnonymous;
    }

    public String getApiKey(String key)
    {
        return apiMap.get(key);
    }

    class Builder{
        String projectID;
        boolean isAnonymous;
        String tncUrl = null;
        HashMap<String,String> apiMap;
        long timeToUploadContext;

        public Builder()
        {
            apiMap = new HashMap<>();
        }

        public Builder setProjectID(String id)
        {
            this.projectID = id;
            return this;
        }

        public Builder isAnonymous(boolean isAnon)
        {
            this.isAnonymous = isAnon;
            return this;
        }

        public Builder termsAndServicesRequired(String tncUrl)
        {
            this.tncUrl = tncUrl;
            return this;
        }

        public Builder addAPI(String apiName, String apiKey)
        {
            apiMap.put(apiName, apiKey);
            return this;
        }

        public Builder timeToUploadContext(int timeToUploadContext){
            this.timeToUploadContext = timeToUploadContext;
            return this;
        }

        public ConciergeConfiguration build()
        {
            return new ConciergeConfiguration(this);
        }
    }

}
