package com.flybits.concierge

import android.content.Intent
import android.media.session.PlaybackState
import android.net.Uri
import android.os.Bundle
import android.support.v4.content.ContextCompat
import android.support.v4.media.session.MediaControllerCompat
import android.support.v4.media.session.PlaybackStateCompat
import android.support.v4.view.ViewPager
import android.text.SpannableString
import android.text.TextUtils
import android.text.style.UnderlineSpan
import android.view.LayoutInflater
import android.view.View
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.TextView
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DecodeFormat
import com.bumptech.glide.request.RequestOptions
import com.flybits.android.kernel.ContentAnalytics
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.activities.EventDetailsActivity
import com.flybits.concierge.activities.ImageViewerActivity
import com.flybits.concierge.activities.VideoActivity
import com.flybits.concierge.activities.VideoPlaylistActivity
import com.flybits.concierge.adapters.ArticleAdapter
import com.flybits.concierge.fragments.ScheduleSegmentFragment
import com.flybits.concierge.models.*
import com.flybits.concierge.services.AudioService
import java.net.URL
import java.util.*

class FlybitsViewPopulator(private val contentAnalytics: ContentAnalytics) {

    fun populateArticleViews(article: Article, contentId: String, headerView: ImageView, titleView: TextView, sourceView: TextView, descriptionView: TextView, openLinkView: View) {
        val options = RequestOptions().format(DecodeFormat.PREFER_RGB_565)

        Glide.with(headerView.context)
                .load(article.imageURL)
                .apply(options)
                .into(headerView)

        titleView.text = article.title.value
        sourceView.visibility = if (!TextUtils.isEmpty(article.sourceName)) View.VISIBLE else View.INVISIBLE
        sourceView.text = article.sourceName
        descriptionView.text = article.description.value

        openLinkView.setOnClickListener { v ->
            contentAnalytics.trackEngaged(contentId, System.currentTimeMillis())
            val url = article.url
            val intent = Intent(Intent.ACTION_VIEW)
            intent.data = Uri.parse(url)
            v.context.startActivity(intent)
        }
    }

    fun populateArticlesViews(data: Articles, viewPager: ViewPager, singleView: View, imgHeader: ImageView
                              , txtTitle: TextView, txtSource: TextView, txtDescription: TextView, thisView: View){
        viewPager.visibility = View.GONE
        singleView.visibility = View.GONE
        val articleList = data.articles.list

        if (articleList.size > 1) {
            viewPager.visibility = View.VISIBLE
            val adapter = ArticleAdapter(this)
            viewPager.adapter = adapter
            adapter.setData(data.articles.list, data.content.id)
        } else if (articleList.size == 1) {
            singleView.visibility = View.VISIBLE
            populateArticleViews(articleList[0], data.content.id, imgHeader, txtTitle, txtSource, txtDescription, thisView)
        }
    }

    fun populateAudioViews(audio: Audio, txtTitle: TextView, txtDescription: TextView, txtDuration: TextView, imgImage: ImageView) {
        txtTitle.text = audio.title.value
        txtDescription.text = audio.description.value
        txtDuration.visibility = if (!TextUtils.isEmpty(audio.duration)) View.VISIBLE else View.GONE
        txtDuration.text = audio.duration

        val options = RequestOptions().format(DecodeFormat.PREFER_RGB_565)

        Glide.with(imgImage.context)
                .load(audio.imageURL)
                .apply(options)
                .into(imgImage)
    }

    fun populateAudiosViews(data: Audios, thisView: LinearLayout, conciergeFragment: ConciergeFragment){
        val audioList = data.audios.list

        thisView.removeAllViews()

        for (audio in audioList) {
            val view = LayoutInflater.from(thisView.context).inflate(R.layout.flybits_con_item_template_audio, thisView, false)

            val imgImage = view.findViewById<ImageView>(R.id.flybits_con_audio_imgImage)
            val imgMediaStatusPlayPause = view.findViewById<ImageView>(R.id.flybits_con_audio_imgMediaStatusPlayPause)
            val imgMediaStatusStop = view.findViewById<ImageView>(R.id.flybits_con_audio_imgMediaStatusStop)
            val txtTitle = view.findViewById<TextView>(R.id.flybits_con_audio_txtTitle)
            val txtDescription = view.findViewById<TextView>(R.id.flybits_con_audio_txtDescription)
            val txtDuration = view.findViewById<TextView>(R.id.flybits_con_audio_txtDuration)

            // populate views
            populateAudioViews(audio, txtTitle, txtDescription, txtDuration, imgImage)

            val playingURL = if (conciergeFragment.currentAudio != null && conciergeFragment.currentAudio.sourceURL != null) conciergeFragment.currentAudio.sourceURL else AudioService.getPlayingURL()
            val playingState = AudioService.getPlayingState()

            if (playingURL != null && playingURL.contentEquals(audio.sourceURL)) {
                if (playingState == PlaybackState.STATE_NONE || playingState == PlaybackState.STATE_STOPPED) {
                    imgMediaStatusPlayPause.setImageResource(R.drawable.ic_baseline_play_arrow_24px)
                } else if (playingState == PlaybackState.STATE_PAUSED) {
                    imgMediaStatusPlayPause.setImageResource(R.drawable.ic_baseline_play_arrow_24px)
                } else if (playingState == PlaybackState.STATE_PLAYING) {
                    imgMediaStatusStop.visibility = View.VISIBLE
                    imgMediaStatusPlayPause.setImageResource(R.drawable.ic_baseline_pause_24px)
                }
            }

            conciergeFragment.registerMediaControllerCallback(object : MediaControllerCompat.Callback() {
                override fun onPlaybackStateChanged(state: PlaybackStateCompat) {
                    super.onPlaybackStateChanged(state)
                    try {
                        when (state.state) {
                            PlaybackState.STATE_NONE -> imgMediaStatusPlayPause.setImageResource(R.drawable.ic_baseline_play_arrow_24px)
                            PlaybackState.STATE_STOPPED -> imgMediaStatusPlayPause.setImageResource(R.drawable.ic_baseline_play_arrow_24px)
                            PlaybackState.STATE_PLAYING -> imgMediaStatusPlayPause.setImageResource(R.drawable.ic_baseline_pause_24px)
                            PlaybackState.STATE_PAUSED -> imgMediaStatusPlayPause.setImageResource(R.drawable.ic_baseline_play_arrow_24px)
                        }
                    } catch (e: Exception) {
                        Logger.exception("AudiosViewHolder.bindData()", e)
                    }

                }
            })

            imgMediaStatusPlayPause.setOnClickListener {
                contentAnalytics.trackEngaged(data.content.id, System.currentTimeMillis())
                if (conciergeFragment.currentAudio != null && conciergeFragment.currentAudio.sourceURL != null && conciergeFragment.currentAudio.sourceURL == audio.sourceURL && conciergeFragment.audioState == PlaybackState.STATE_PLAYING) {
                    conciergeFragment.pauseAudio()
                } else if (conciergeFragment.currentAudio != null && conciergeFragment.currentAudio.sourceURL != null && conciergeFragment.currentAudio.sourceURL == audio.sourceURL && conciergeFragment.audioState == PlaybackState.STATE_PAUSED) {
                    conciergeFragment.resumeAudio()
                } else if (conciergeFragment.audioState == PlaybackState.STATE_NONE || conciergeFragment.audioState == PlaybackState.STATE_STOPPED) {
                    conciergeFragment.startAudio(audio)
                }
            }

            imgMediaStatusStop.setOnClickListener {
                contentAnalytics.trackEngaged(data.content.id, System.currentTimeMillis())
                conciergeFragment.stopAudio()
            }

            thisView.addView(view)
        }
    }

    fun populateEventViews(event: Event, flybitsNavigator: FlybitsNavigator?, titleView: TextView?, locationView: TextView, dateView: TextView, imgHeader: ImageView, expandView: View?, addressView: TextView?, ctaView: TextView?, phoneView: TextView?, descriptionView: TextView?) {
        val options = RequestOptions().centerCrop().format(DecodeFormat.PREFER_RGB_565)

        Glide.with(imgHeader.context)
                .load(event.coverPhotoURL)
                .apply(options)
                .into(imgHeader)

        titleView?.text = if (!TextUtils.isEmpty(event.title.value)) event.title.value else ""
        locationView.text = if (!TextUtils.isEmpty(event.venueDescription.value)) event.venueDescription.value else ""
        dateView.text = Utils.getDateRange(event.getStartDate(), event.getEndDate())

        if (expandView != null && flybitsNavigator != null) {
            expandView.setOnClickListener {
                val extras = Bundle()
                extras.putParcelable(EventDetailsActivity.EXTRA_EVENT_DETAILS, event)
                flybitsNavigator.openActivity(EventDetailsActivity::class.java, extras)
            }
        }

        if (addressView != null) {
            if (descriptionView != null) {
                val content = SpannableString(event.address)
                content.setSpan(UnderlineSpan(), 0, content.length, 0)
                addressView.text = content
            } else {
                addressView.text = event.address
            }
        }

        if (ctaView != null) {
            ctaView.visibility = if (!TextUtils.isEmpty(event.url)) View.VISIBLE else View.INVISIBLE

            ctaView.setOnClickListener { v ->
                val url = event.url
                val intent = Intent(Intent.ACTION_VIEW)
                intent.data = Uri.parse(url)
                v.context.startActivity(intent)
            }
        }

        if (phoneView != null) {
            if (!TextUtils.isEmpty(event.phoneNumber)) {
                if (descriptionView != null) {
                    val content = SpannableString(event.phoneNumber)
                    content.setSpan(UnderlineSpan(), 0, content.length, 0)
                    phoneView.text = content
                } else {
                    phoneView.text = event.phoneNumber
                }
            } else {
                phoneView.setText(android.R.string.unknownName)
            }
        }

        descriptionView?.text = event.description.value
    }

    fun populateEventViews(event: Event, flybitsNavigator: FlybitsNavigator, titleView: TextView, locationView: TextView, dateView: TextView, imgHeader: ImageView, expandView: View) {
        populateEventViews(event, flybitsNavigator, titleView, locationView, dateView, imgHeader, expandView, null, null, null)
    }

    fun populateEventViews(event: Event, flybitsNavigator: FlybitsNavigator, titleView: TextView, locationView: TextView, dateView: TextView, imgHeader: ImageView, expandView: View, addressView: TextView?, ctaView: TextView?, phoneView: TextView?) {
        populateEventViews(event, flybitsNavigator, titleView, locationView, dateView, imgHeader, expandView, addressView, ctaView, phoneView, null)
    }

    fun populateEventsViews(data: Events, flybitsNavigator: FlybitsNavigator, thisView: LinearLayout){
        thisView.removeAllViews()

        for (event in data.events.list) {
            val eventView = LayoutInflater.from(thisView.context).inflate(R.layout.flybits_con_item_template_event_item, thisView, false)

            val imgHeader = eventView.findViewById<View>(R.id.flybits_con_events_item_imgHeader) as ImageView
            val txtTitle = eventView.findViewById<View>(R.id.flybits_con_events_item_txtTitle) as TextView
            val txtLocation = eventView.findViewById<View>(R.id.flybits_con_events_item_txtLocation) as TextView
            val txtDate = eventView.findViewById<View>(R.id.flybits_con_events_item_txtDate) as TextView

            populateEventViews(event, flybitsNavigator, txtTitle, txtLocation, txtDate, imgHeader, eventView)

            thisView.addView(eventView)
        }
    }

    fun populateLinkViews(link: Link, parentContentId: String, txtHeading: TextView, txtBody: TextView, linkView: View) {
        txtHeading.text = if (!TextUtils.isEmpty(link.title.value)) link.title.value else ""

        txtBody.text = if (!TextUtils.isEmpty(link.description.value)) link.description.value else ""
        txtBody.visibility = if (TextUtils.isEmpty(link.description.value)) View.GONE else View.VISIBLE

        linkView.setOnClickListener { v ->
            contentAnalytics.trackEngaged(parentContentId, System.currentTimeMillis())
            val url = link.url
            val intent = Intent(Intent.ACTION_VIEW)
            intent.data = Uri.parse(url)
            v.context.startActivity(intent)
        }
    }

    fun populateLinksViews(data: Links, thisView: LinearLayout){

        // remove all views
        thisView.removeAllViews()

        var shouldAddRule = false

        for (link in data.links.list) {
            val linkView = LayoutInflater.from(thisView.context).inflate(R.layout.flybits_con_item_template_link, thisView, false)

            val txtHeading:TextView = linkView.findViewById(R.id.flybits_con_item_template_link_txtTitle)
            val txtBody:TextView = linkView.findViewById(R.id.flybits_con_item_template_link_txtBody)
            val hr: View = linkView.findViewById(R.id.flybits_con_item_template_link_viewRule)

            hr.visibility = if (shouldAddRule) View.VISIBLE else View.GONE
            shouldAddRule = true

            populateLinkViews(link, data.content.id, txtHeading, txtBody, linkView)

            thisView.addView(linkView)
        }
    }

    fun populateOnboardingViews(onboarding: Onboarding, titleTextView: TextView) {
        var onBoardingTitle: String? = null

        for (page in onboarding.onboardingPages.list) {
            if (!TextUtils.isEmpty(page.title.value)) {
                onBoardingTitle = page.title.value
                break
            }
        }

        val displayedTitle = String.format(Locale.getDefault(), titleTextView.context.getString(R.string.flybits_con_item_template_on_boarding_title_format), onBoardingTitle)

        titleTextView.text = displayedTitle
    }

    fun populateOnboardingPageViews(onboardingPage: OnboardingPage, titleTextView: TextView) {
        val onBoardingTitle = onboardingPage.title.value

        val displayedTitle = String.format(Locale.getDefault(), titleTextView.context.getString(R.string.flybits_con_item_template_on_boarding_title_format), onBoardingTitle)

        titleTextView.text = displayedTitle
    }

    fun populateOnboardingPageViews(onboardingPage: OnboardingPage, titleTextView: TextView, bodyTextView: TextView,logoImageView: ImageView){
        titleTextView.text = onboardingPage.title.value
        bodyTextView.text = onboardingPage.description.value

        val options = RequestOptions().format(DecodeFormat.PREFER_RGB_565)

        Glide.with(logoImageView.context)
                .load(onboardingPage.imageUrl)
                .apply(options)
                .into(logoImageView)
    }

    fun populateScheduleSegmentViews(flybitsNavigator: FlybitsNavigator, data: ScheduleSegment, clickableArea: View, timeRangeView: TextView
                                     , titleView: TextView, descriptionView: TextView
                                     , bulletOutView: ImageView, bulletInView: ImageView){

        clickableArea.setOnClickListener {
           //TODO: Fix this using annotations, even passing content in BaseTemplate will cause TransactionTooLargeException
            // contentAnalytics.trackEngaged(data.content.id, System.currentTimeMillis())
            val fragment = ScheduleSegmentFragment.newInstance(data)
            flybitsNavigator.openFragment(fragment)
        }

        populateScheduleSegmentViews(data, timeRangeView, titleView, descriptionView, bulletOutView, bulletInView, clickableArea)
    }

    fun populateScheduleSegmentViews(scheduleSegment: ScheduleSegment, timeRangeView: TextView
                                     , titleView: TextView, descriptionView: TextView
                                     , bulletOutView: ImageView, bulletInView: ImageView
                                     , containerView: View) {
        val range = Utils.getTimeRange(scheduleSegment.startDayTime, scheduleSegment.endDayTime)
        timeRangeView.text = range
        titleView.text = scheduleSegment.title.value
        descriptionView.text = scheduleSegment.description.value

        val today = Calendar.getInstance()

        val isCurrent = today.time.after(scheduleSegment.startDayTime) && today.time.before(scheduleSegment.endDayTime)
        val isPast = today.time.after(scheduleSegment.endDayTime)

        bulletOutView.setColorFilter(ContextCompat.getColor(bulletOutView.context, if (isCurrent) R.color.flybits_con_schedule_bullet_out_selected else R.color.flybits_con_schedule_bullet_out), android.graphics.PorterDuff.Mode.MULTIPLY)
        bulletInView.setColorFilter(ContextCompat.getColor(bulletInView.context, if (isCurrent) R.color.flybits_con_schedule_bullet_in_selected else R.color.flybits_con_schedule_bullet_in), android.graphics.PorterDuff.Mode.MULTIPLY)

        containerView.setBackgroundColor(ContextCompat.getColor(containerView.context, if (isPast) R.color.flybits_con_schedule_segment_item_background_past else if (isCurrent) R.color.flybits_con_schedule_segment_item_background_selected else R.color.flybits_con_schedule_segment_item_background))

        val textColor = ContextCompat.getColor(titleView.context, if (isCurrent) R.color.flybits_white else R.color.flybits_con_body_text_color)
        timeRangeView.setTextColor(textColor)
        titleView.setTextColor(textColor)
        descriptionView.setTextColor(textColor)
    }

    fun populateVideosViews(data: Videos, flybitsNavigator: FlybitsNavigator, thisView: View
                            , singleContainer: View, doubleContainer: View, tripleContainer: View
                            , singlePlaceHolderViews: Array<ImageView>
                            , doublePlaceHolderViews: Array<ImageView>
                            , triplePlaceHolderViews: Array<ImageView>, callToActionView: View){
        val videos = data.videos.list
        videos.forEach { it.content = data.content }

        tripleContainer.visibility = if (videos.size > 2) View.VISIBLE else View.GONE
        doubleContainer.visibility = if (videos.size == 2) View.VISIBLE else View.GONE
        singleContainer.visibility = if (videos.size == 1) View.VISIBLE else View.GONE
        callToActionView.visibility = if (videos.size > 1) View.VISIBLE else View.GONE

        val placeHolderViews: Array<ImageView>

        placeHolderViews = when (videos.size) {
            0 -> return
            1 -> {
                val titleTextView : TextView = thisView.findViewById(R.id.flybits_con_item_template_videos_txtTitle)
                val durationTextView: TextView = thisView.findViewById(R.id.flybits_con_item_template_videos_txtDuration)
                populateVideoViews(videos[0], data.content.id ,flybitsNavigator, titleTextView, durationTextView, singlePlaceHolderViews[0], thisView)
                return
            }
            2 -> doublePlaceHolderViews
            else -> triplePlaceHolderViews
        }

        for (i in placeHolderViews.indices) {
            val view = placeHolderViews[i]
            if (i < videos.size) {
                val video = videos.get(i)

                populateVideoViews(video, data.content.id, flybitsNavigator, view, view)
            } else {
                view.setImageDrawable(ContextCompat.getDrawable(view.context, android.R.drawable.bottom_bar))
                view.setOnClickListener(null)
            }
        }

        callToActionView.setOnClickListener{
            contentAnalytics.trackEngaged(data.content.id, System.currentTimeMillis())
            val extras = Bundle()
            extras.putParcelableArrayList(VideoPlaylistActivity.EXTRA_VIDEO, videos)
            flybitsNavigator.openActivity(VideoPlaylistActivity::class.java, extras)
        }
    }

    fun populateVideoViews(video: Video, parentContentId: String, flybitsNavigator: FlybitsNavigator
                           , placeholderImageView: ImageView, callToActionView: View) {
        populateVideoViews(video, parentContentId, flybitsNavigator, null, null, placeholderImageView, callToActionView)
    }

    fun populateVideoViews(video: Video, parentContentId: String, fragmentSwitcher: FlybitsNavigator
                           , titleTextView: TextView?, durationTextView: TextView?
                           , placeholderImageView: ImageView, callToActionView: View) {
        titleTextView?.text = video.title.value

        if (durationTextView != null) {
            durationTextView.visibility = if (!TextUtils.isEmpty(video.duration)) View.VISIBLE else View.INVISIBLE
            durationTextView.text = video.duration
        }

        val options = RequestOptions().format(DecodeFormat.PREFER_RGB_565)

        Glide.with(placeholderImageView.context)
                .load(video.imageURL)
                .apply(options)
                .into(placeholderImageView)

        callToActionView.setOnClickListener(View.OnClickListener { v ->
            contentAnalytics.trackEngaged(parentContentId, System.currentTimeMillis())
            // YouTube
            try {
                val videoUrl = URL(video.url)
                videoUrl.host

                var videoId: String? = null

                when (videoUrl.host) {
                    "www.youtube.com" -> {
                        val map = Utils.splitQuery(videoUrl)
                        videoId = map["v"]
                    }
                    "youtu.be" -> {
                        val path = videoUrl.path
                        if (!TextUtils.isEmpty(path) && path.length > 2) {
                            videoId = videoUrl.path
                                    .substring(1)
                        }
                    }
                }

                if (!TextUtils.isEmpty(videoId)) {
                    video.watchYoutubeVideo(v.context, videoId)
                    return@OnClickListener
                }
            } catch (e: Exception) {
                Logger.exception(Video::class.java.simpleName, e)
            }

            // Server
            val extras = Bundle()
            extras.putParcelable(VideoActivity.EXTRA_VIDEO, video)
            fragmentSwitcher.openActivity(VideoActivity::class.java, extras)
        })
    }

    fun populateImagesViews(images: Images, flybitsNavigator: FlybitsNavigator, thisView: View
                            , singleView: ImageView, doubleView: View, multipleView: View
                            , doubleImages: Array<ImageView>, multipleImages: Array<ImageView>
                            , currentBoundImage: Images){


        val onClickListener = View.OnClickListener { view ->
            contentAnalytics.trackEngaged(images.content.id, System.currentTimeMillis())

            val extras = Bundle()
            extras.putParcelable(ImageViewerActivity.EXTRA_IMAGES, currentBoundImage.images)

            if (view == singleView) {
                extras.putInt(ImageViewerActivity.EXTRA_INDEX, 0)
                flybitsNavigator.openActivity(ImageViewerActivity::class.java, extras)
            } else {
                for (i in multipleImages.indices) {
                    if (i < doubleImages.size && view == doubleImages[i] || view == multipleImages[i]) {
                        extras.putInt(ImageViewerActivity.EXTRA_INDEX, i)
                        flybitsNavigator.openActivity(ImageViewerActivity::class.java, extras)
                        break
                    }
                }
            }
        }

        val imageViews: Array<ImageView>

        singleView.visibility = View.GONE
        doubleView.visibility = View.GONE
        multipleView.visibility = View.GONE

        val options = RequestOptions().centerCrop().format(DecodeFormat.PREFER_RGB_565)

        if (images.images.list.size == 1) {
            val image = images.images.list.get(0)
            singleView.visibility = View.VISIBLE

            Glide.with(thisView.context)
                    .load(image.url)
                    .apply(options)
                    .into(singleView)
            singleView.setOnClickListener(onClickListener)

            return
        } else {
            when (images.images.list.size) {
                2 -> {
                    imageViews = doubleImages
                    doubleView.visibility = View.VISIBLE
                }
                else -> {
                    imageViews = multipleImages
                    multipleView.visibility = View.VISIBLE
                }
            }
        }

        for (i in imageViews.indices) {
            val imgView = imageViews[i]
            Glide.with(thisView.context)
                    .load(images.images.list.get(i).url)
                    .apply(options)
                    .into(imgView)
            imgView.setOnClickListener(onClickListener)
        }
    }
}