package com.flybits.concierge.repository

import android.arch.lifecycle.MutableLiveData
import android.content.Context
import android.os.Handler
import android.os.Looper
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.models.Project
import com.flybits.concierge.models.CategorySettings
import java.util.concurrent.Executors

class CategorySettingsRepository(private val categoryDao: CategoryDao, private val context: Context) {

    data class CategorySettingsResponse(val categorySettings: CategorySettings? = null, val exception: FlybitsException? = null)

    private val executorService = Executors.newSingleThreadExecutor()

    companion object {
        const val TYPE_CATEGORY_SETTINGS = "category-settings"
    }

    //Handler is optional here, only added for testing purposes.
    fun getCategorySettingsRemoteLive(handler: Handler = Handler(Looper.getMainLooper())): MutableLiveData<CategorySettingsResponse> {
        val categoryLiveData: MutableLiveData<CategorySettingsResponse> = MutableLiveData()
        getCategorySettingsRemote(object: ObjectResultCallback<CategorySettings>{
            override fun onSuccess(item: CategorySettings) {
                categoryLiveData.value = CategorySettingsResponse(categorySettings = item)
            }

            override fun onException(exception: FlybitsException?) {
                categoryLiveData.value = CategorySettingsResponse(exception = exception)
            }

        }, handler)
        return categoryLiveData
    }

    //we can call this just to sync with server with null parameter
    fun getCategorySettingsRemote(callback: ObjectResultCallback<CategorySettings>? = null, handler: Handler = Handler(Looper.getMainLooper())){
        executorService.execute{
            Project.getSettings(TYPE_CATEGORY_SETTINGS, context.applicationContext, CategorySettings::class.java, object: ObjectResultCallback<CategorySettings>{

                override fun onSuccess(item: CategorySettings?) {
                    if (item != null){
                       executorService.execute { categoryDao.deleteAndInsertTransaction(item.categories) }
                    }
                    callback?.onSuccess(item)
                }

                override fun onException(exception: FlybitsException?) {
                    callback?.onException(exception)
                }

            }, handler)
        }
    }

    fun getCategorySettingsLocal() = categoryDao.getAll()
}
