package com.flybits.concierge.fragments

import android.arch.lifecycle.Observer
import android.arch.lifecycle.ViewModelProviders
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.Bundle
import android.support.design.widget.Snackbar
import android.support.v4.app.FragmentActivity
import android.support.v4.widget.SwipeRefreshLayout
import android.support.v7.widget.LinearLayoutManager
import android.support.v7.widget.RecyclerView
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup

import com.afollestad.materialdialogs.MaterialDialog
import com.flybits.android.kernel.models.Content
import com.flybits.android.push.analytics.PushAnalytics
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.*
import com.flybits.concierge.activities.ConciergeActivity
import com.flybits.concierge.activities.ConciergePopupActivity
import com.flybits.concierge.adapters.FeedAdapter
import com.flybits.concierge.models.BaseTemplate
import com.flybits.concierge.viewmodels.CategoryViewModel
import com.flybits.concierge.viewmodels.CategoryViewModelFactory
import com.flybits.concierge.viewmodels.TwitterViewModel

class CategoryFragment : BasePagerFragment(), CategoryViewModel.FeedErrorDisplayer {

    companion object {
        fun newInstance(category: String): CategoryFragment {
            val bundle = Bundle()
            bundle.putString("category", category)
            val fragment = CategoryFragment()
            fragment.arguments = bundle
            return fragment
        }
    }

    private var recyclerView: RecyclerView? = null
    private var categoryViewModel: CategoryViewModel? = null
    private var feedAdapter: FeedAdapter? = null
    private var noDataView: View? = null
    private var refreshReceiver: BroadcastReceiver? = null
    private var swipeRefreshLayout: SwipeRefreshLayout? = null

    override fun titleRes(): Int {
        return R.string.flybits_con_fragment_title_feed
    }

    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View? {
        // Inflate the layout for this fragment
        return inflater.inflate(R.layout.flybits_con_fragment_category, container, false)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        Logger.d("Category: ${arguments!!.getString("category","")} fragment view created!")

//        view.findViewById<View>(R.id.flybits_con_fragment_feed_settings_button).setOnClickListener {
//            val parentFragment = parentFragment
//            if (parentFragment != null && parentFragment is ConciergeFragment) {
//                parentFragment.startSettingsActivity()
//            }
//        }

        val linearLayoutManager = LinearLayoutManager(context)
        recyclerView = view.findViewById(R.id.flybits_con_fragment_feed_lytRecycler)
        recyclerView?.layoutManager = linearLayoutManager
        var conciergeFragment: ConciergeFragment? = null
        fragmentManager?.fragments?.forEach {
            if (it is ConciergeFragment){
                conciergeFragment = it
            }
        }

        feedAdapter = FeedAdapter(conciergeFragment!!, conciergeFragment!!)
        recyclerView?.adapter = feedAdapter
        recyclerView?.setHasFixedSize(false)
        recyclerView?.addOnScrollListener(object : RecyclerView.OnScrollListener() {
            override fun onScrolled(recyclerView: RecyclerView?, dx: Int, dy: Int) {
                val firstVisibleItem = linearLayoutManager.findFirstCompletelyVisibleItemPosition()
                val lastVisibleItem = linearLayoutManager.findLastCompletelyVisibleItemPosition()
                if (firstVisibleItem != RecyclerView.NO_POSITION && lastVisibleItem != RecyclerView.NO_POSITION){
                    val items = feedAdapter?.currentList?.subList(firstVisibleItem, lastVisibleItem+1)
                    items?.filter { it != null }?.let {
                        categoryViewModel?.onScroll(it)
                    }
                }
                super.onScrolled(recyclerView, dx, dy)
            }
        })

        swipeRefreshLayout = view.findViewById(R.id.flybits_con_swipe_ref_layout)
        swipeRefreshLayout?.setOnRefreshListener {
            Logger.d(CategoryFragment::class.java.simpleName + ": onRefresh()")
            categoryViewModel?.refresh(null)

            if (context != null) {
                val twitterViewModel = ViewModelProviders
                        .of(context as FragmentActivity)
                        .get(TwitterViewModel::class.java)

                twitterViewModel.clearTweetLists()
            }
        }

        noDataView = view.findViewById(R.id.flybits_con_fragment_feed_lytNoData)

        categoryViewModel = ViewModelProviders.of(this, CategoryViewModelFactory(context!!
                , conciergeFragment as FlybitsViewProviderGetter, arguments!!.getString("category")))
                .get(CategoryViewModel::class.java)

        swipeRefreshLayout?.isRefreshing = true
        categoryViewModel?.feedContent
                ?.observe(this, Observer { baseTemplates ->
                    Logger.d("Category: ${arguments!!.getString("category","")}, Got data! data: $baseTemplates")
                    feedAdapter?.submitList(baseTemplates)
                    val baseTemplatesEmpty = baseTemplates?.isEmpty() ?: false
                    recyclerView?.visibility = if (baseTemplatesEmpty) View.GONE else View.VISIBLE
                    noDataView?.visibility = if (baseTemplatesEmpty) View.VISIBLE else View.GONE
                    swipeRefreshLayout?.isRefreshing = false
                })
        categoryViewModel?.setFeedErrorDisplayer(this)

        refreshReceiver = object : BroadcastReceiver() {
            override fun onReceive(context: Context, intent: Intent) {
                if (feedAdapter != null) {
                    try {
                        feedAdapter?.notifyDataSetChanged()
                    } catch (e: Exception) {
                        Logger.exception("CategoryFragment.RefreshReceiver.onReceiver()", e)
                    }

                }
            }
        }

        val activity = activity
        val metaDataId = activity?.intent?.getStringExtra(ConciergeConstants.PUSH_CONTENT_ID)
        val url = activity?.intent?.getStringExtra(ConciergeConstants.PUSH_CONTENT_URL)

        //La Analytics
        val pushId = activity?.intent?.getStringExtra(ConciergeConstants.PUSH_ID)
        if (pushId != null) context?.let{
            PushAnalytics(it).trackEngaged(pushId, System.currentTimeMillis())
        }

        if (metaDataId != null) {
            categoryViewModel?.connectAndDownLoad(activity, metaDataId, object : ObjectResultCallback<Content> {

                override fun onSuccess(item: Content) {
                    var conciergeFragment: ConciergeFragment? = null
                    fragmentManager?.fragments?.forEach {
                        if (it is ConciergeFragment){
                            conciergeFragment = it
                        }
                    }
                    if (conciergeFragment != null) {
                        val flybitsViewProviderGetter = conciergeFragment as FlybitsViewProviderGetter?
                        val flybitsNavigator = conciergeFragment as FlybitsNavigator
                        val flybitsViewProvider = flybitsViewProviderGetter?.getFlybitsViewProvider(item.type)
                        if (flybitsViewProvider != null) {
                            val baseTemplate = BaseTemplate.fromContent(flybitsViewProvider.getClassType(), item, context)
                            if (baseTemplate != null) {
                                flybitsViewProvider.onPushOpened(baseTemplate, flybitsNavigator)
                            }
                        }
                    }
                }

                override fun onException(e: FlybitsException) {
                    //Make sure activity isn't null once the callback is invoked
                    val currentActivity = getActivity()
                    if (currentActivity != null) {
                        MaterialDialog.Builder(currentActivity)
                                .title(R.string.flybits_con_error)
                                .content(R.string.flybits_con_could_not_load)
                                .positiveText(android.R.string.ok)
                                .dismissListener { currentActivity.finish() }.show()
                    }

                    Logger.exception(CategoryFragment::class.java.simpleName, e)
                }
            })
        } else if (url != null) {
            (parentFragment as FlybitsNavigator).openUrl(url)
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        try {
            val activity = activity
            activity?.registerReceiver(refreshReceiver, IntentFilter(ConciergeConstants.BROADCAST_AUDIO_STATE))
        } catch (e: Exception) {
            Logger.exception("CategoryFragment.onCreate()", e)
        }

    }

    override fun onDestroy() {
        super.onDestroy()
        //Receiver may already be unregistered or may have never been registered in that case catch the exception
        try {
            val activity = activity
            activity?.unregisterReceiver(refreshReceiver)
        } catch (e: Exception) {
            Logger.exception("CategoryFragment.onDestroy()", e)
        }

    }

    override fun onError(err: String) {
        Logger.d("CategoryFragment: onAuthenticationError() err: $err")
        val activity = activity
        val v = view
        /*The instanceof check is done to make sure that the user is still within
        * the concierge feature at the time the toast is displayed.*/
        if (activity != null && (activity is ConciergeActivity || activity is ConciergePopupActivity) && v != null) {
            Snackbar.make(v, err, Snackbar.LENGTH_LONG).show()
        }
    }
}
