package com.flybits.commons.library.api.results

import android.content.Context
import android.os.Handler
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.http.RequestStatus
import com.flybits.commons.library.models.internal.PagedResponse
import com.flybits.commons.library.models.internal.Pagination
import com.flybits.commons.library.models.internal.QueryParameters
import com.flybits.commons.library.models.internal.Result
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors

/**
 * The [PagedResult] class is used to query the server when the response is excepted to have
 * a pagination object. Through this class, the application will be able to call the [getMore]
 * method automatically in order to retrieve the next set of results.
 *
 * @param context The context of the application.
 * @param filters The filter options used to retrieve data from the server.
 * @param callback The [PagedResultCallback] that indicates whether or not the network
 *                 request was successful or not, if successful a list of objects will be returned.
 * @param handler The handler that allows methods to be run in the UI.
 * @param service The [ExecutorService] that is used to run the network request thread.
 */
abstract class PagedResult<T>(val context: Context,
                              private val filters: QueryParameters,
                              val callback: PagedResultCallback<T>? = null,
                              val handler: Handler? = null,
                              service: ExecutorService = Executors.newSingleThreadExecutor()) : FlybitsResult(service) {

    private var currentOffset = filters.offset
    private var limit = filters.limit
    private var totalItems : Long = 0

    /**
     * Makes a network request to get the next set of items if there are more items to be retrieved.
     *
     * @param callback The [PagedResultCallback] that indicates whether or not more items where
     *                 able to be retrieved.
     */
    fun getMore(callback : PagedResultCallback<T>? = null){

        if (!hasMore()){
            val requestCallback = callback ?: this.callback
            if (requestCallback != null) {
                handler?.post {
                    requestCallback.onLoadedAllItems()
                } ?: requestCallback.onLoadedAllItems()
            }
            return
        }

        filters.setPaging(limit, currentOffset)

        if (callback == null) {
            getMoreRequest(filters, object : PagedResultCallback<T> {
                override fun onSuccess(items: ArrayList<T>, pagination: Pagination) {
                    setSuccess(items, pagination)
                }

                override fun onException(exception: FlybitsException) {
                    setFailed(exception)
                }

                override fun onLoadedAllItems() {}
            })
        }else{
            getMoreRequest(filters, callback)
        }
    }

    /**
     * Set the [Result] of the network request that contains the list of objects returned from
     * the network request.
     *
     * @param result The [Result] that contains a [PagedResponse] object which will
     *               contain the list of items as well as the [Pagination] object that
     *               indicates paging information about this request.
     */
    fun setResult(result : Result<PagedResponse<T>>){

        if (result.status == RequestStatus.COMPLETED){
            if (result.result == null){
                setFailed(FlybitsException("There was an issue parsing the response from the server. This is unexpected, please contact support@flybits.com for support."))
            }else setSuccess(result.result.items, result.result.pagination)
        }else setFailed(result.exception)
    }

    /**
     * Responsible for retrieving more items of the desired query. Each query should implement it's
     * own [getMoreRequest] method as each query may have
     * it's own logic for retrieving items.
     *
     * @param filter The [QueryParameters], if any, that that indicate any additional
     *                   filters used for this network request.
     * @param callback The [PagedResultCallback] that indicates whether or not more items where
     *                 able to be retrieved.
     */
    abstract fun <K : QueryParameters> getMoreRequest(filter : K,
                                                      callback : PagedResultCallback<T>)

    internal fun setFailed(e: FlybitsException): Boolean {
        return if (failed()) {
            if (callback != null) {
                handler?.post { callback.onException(e) } ?: callback.onException(e)
            }
            true
        } else false
    }

    internal fun setSuccess(items : ArrayList<T>, pagination : Pagination){
        totalItems = pagination.totalRecords
        currentOffset = pagination.offset + items.size

        if (success() && callback != null)
            handler?.post{postSuccess(items, pagination)} ?: postSuccess(items, pagination)
    }

    private fun hasMore() = limit < totalItems && currentOffset < totalItems

    private fun postSuccess(items : ArrayList<T>, pagination : Pagination) {
        if (callback != null){
            callback.onSuccess(items, pagination)
            if (!hasMore())
                callback.onLoadedAllItems()
        }
    }
}