package com.flybits.commons.library

import android.content.Context
import android.content.SharedPreferences
import com.flybits.commons.library.api.FlybitsManager
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.utils.Utilities
import com.flybits.internal.db.UserDAO
import kotlinx.coroutines.*
import kotlinx.coroutines.Dispatchers.IO

/**
 * This class is used to get and set all variables that can be shared across the various Flybits'
 * Android SDKs. Currently the items that can be shared between the SDKs are as follows;
 *
 *  * Language Information
 *  * Device Identifier
 *  * User Identifier
 *  * JWT
 *  * Project ID
 *  * Connected IDP
 *
 *  @param sharedPreferences [SharedPreferences] that data will be stored and retrieved from.
 *  @param userDAO [UserDAO] that user data will be retrieved from.
 *
 */
abstract class SharedElements internal constructor(
    protected val sharedPreferences: SharedPreferences, private val userDAO: UserDAO,
    dispatcher:CoroutineDispatcher = newSingleThreadContext("SharedElements$IO")) {

    private var jwtToken : String? = null
    private var languageCodes : String? = null
    private var idp : String? = null
    private var notificationChannelID : String? = null
    private var isPushNotificationEnabled : String? = null
    private var projectID : String? = null
    private var gateway : String? = null
    private var userId : String? = null
    private var locationPermission : String? = null

    internal val job = CoroutineScope(SupervisorJob() + dispatcher)

    companion object {
        internal val TAG = SharedElements::class.java.simpleName

        const val PREF_LANGUAGE_CODES = "com.flybits.language.codes"
        const val PREF_JWT_TOKEN = "com.flybits.jwt.token"
        const val PREF_IDP_CONNECTED = "com.flybits.idp.connected"
        const val PREF_NOTIFICATION_CHANNEL_ID = "com.flybits.idp.channel_id"
        const val PREF_IS_NOTIFICATION_ENABLED = "com.flybits.idp.isPushNotificationEnabled"
        const val PREF_PROJECT_ID = "com.flybits.project.id"
        const val PREF_GATEWAY_URL = "com.flybits.project.url"
        const val PREF_UNIQUE_ID = "com.flybits.device.unique_id"
        const val PREF_USER_ID = "com.flybits.user.id"
        const val PREF_LOCATION_PERMISSION = "com.flybits.location.permission"

        const val FLYBITS_STORAGE_UNENCRYPTED_V0 = "FLYBITS_PREF"        // used before encryption existed
        const val FLYBITS_STORAGE_UNENCRYPTED_V1 = "FLYBITS_PREF_BACKUP" // previously used as a backup from encryption
        const val FLYBITS_STORAGE_UNENCRYPTED_V2 = "FLYBITS_PREF"        // now used as backup from encryption
        const val FLYBITS_STORAGE_ENC_V1 = "flybits_con_storage"
        const val FLYBITS_STORAGE_ENC_V2 = "flybits_secure_storage_v2"
        const val ARG_PROJECT_ID = "flybits_arg_project_id"
    }

    /**
     * Migrate data from previous versions of [SharedElements] to the latest version.
     *
     * @param context Context associated with the Application.
     * @param args Arguments required for migration.
     *
     * @return how many values were migrated successfully.
     */
    fun migrateData(context: Context, args: Map<String, String>): Int {
        clearMemory()
        return performMigration(context, args)
    }

    /**
     * Migrate data from previous versions of [SharedElements] to the latest version.
     *
     * @param context Context associated with the Application.
     * @param args Arguments required for migration.
     *
     * @return how many values were migrated successfully.
     */
    protected abstract fun performMigration(context: Context, args: Map<String, String>): Int

    /**
     * Get the IDP that the application used to connect to Flybits with. If the application is not
     * connected to Flybits then "" empty string will be returned.
     *
     * @return The saved string representation of the IDP used to connect to Flybits with, or "" if
     * the user is not connected to Flybits.
     */
    fun getConnectedIDP() : String {
        if (idp == null){
            idp = getStringVariable(PREF_IDP_CONNECTED, "")
        }
        return idp ?: ""
    }

    /**
     * Get the previously saved [com.flybits.commons.library.models.User.deviceID].
     *
     * @return The saved [com.flybits.commons.library.models.User.deviceID] or "" if no
     * [com.flybits.commons.library.models.User.deviceID] is saved.
     */
    fun getDeviceID(): String {
        val user = userDAO.activeUser
        return if (user != null) {
            user.deviceID
        } else ""
    }

    /**
     * Get the ArrayList representation of the language codes set for this application.
     *
     * @return The ArrayList representation of the language codes set for this application.
     */
    fun getEnabledLanguagesAsArray(): ArrayList<String> {
        val languageCodes = getEnabledLanguagesAsString()
        return Utilities.convertLocalizationStringToList(languageCodes)
    }

    /**
     * Get the String representation of the language codes set for this application.
     *
     * @return The String representation of the language codes set for this application or "" if no
     * language is set.
     */
    fun getEnabledLanguagesAsString() : String {
        if (languageCodes == null){
            languageCodes = getStringVariable(PREF_LANGUAGE_CODES, "")
        }
        return languageCodes?: ""
    }

    /**
     * Get the gateway URL to be used for communicating with the Flybits servers.
     *
     * @return The root URL that all requests should point to.
     */
    fun getGatewayURL() : String {
        if (gateway == null){
            gateway = getStringVariable(PREF_GATEWAY_URL, "")
        }
        return gateway?: ""
    }

    /**
     * Get the unique Flybits Project Identifier associated to your project. This project is set
     * through the [com.flybits.commons.library.api.idps.IDP.getProjectID].
     *
     * @return The saved unique Project identifier which can be retrieved from the
     * [Developer Portal](https://developer.flybits.com)  or "" if no project id.
     */
    fun getProjectID() : String {
        if (projectID == null){
            projectID = getStringVariable(PREF_PROJECT_ID, "")
        }
        return projectID?: ""
    }


    /**
     * Get the Notification Channel ID
     *
     * @return Push Notification channel ID
     */
    fun getNotificationChannel() :String {
        return if (notificationChannelID?.isNotBlank() == true) {
            notificationChannelID ?: ""
        }else  getStringVariable(PREF_NOTIFICATION_CHANNEL_ID,"")
    }


    /**
     * Get the Notification authentication status
     *@return Is the Notification been authorized
     */

    fun getIsPushNotificationEnabled() : Boolean {
        return if (isPushNotificationEnabled?.isNotBlank() == true) {
            isPushNotificationEnabled == "true"
        }else
            getStringVariable(PREF_IS_NOTIFICATION_ENABLED,"true") == "true"
    }

    /**
     * Get the device current location permission as string
     *
     * @return location permission, one of always/inUse/never
     */
    fun getLocationPermission() :String {
        if (locationPermission == null){
            locationPermission = getStringVariable(PREF_LOCATION_PERMISSION, "")
        }
        return locationPermission?: ""
    }


    /**
     * Get the previously saved `JWT` which is obtained when calling
     * [FlybitsManager.connect] the first time.
     *
     * @return The saved `JWT` which is obtained once the application logs into Flybits  or ""
     * if the application has not successfully received a `JWT` token.
     */
    fun getSavedJWTToken() : String {
        if (jwtToken == null){
            jwtToken = getStringVariable(PREF_JWT_TOKEN, "")
        }
        return jwtToken?: ""
    }

    /**
     * Get the previously saved [com.flybits.commons.library.models.User.id].
     *
     * @return The saved [com.flybits.commons.library.models.User.id] or "" if no
     * [com.flybits.commons.library.models.User.id] is saved.
     */
    @Deprecated ("Get the user Id from CommonsDatabase using CommonDatabase.get(context).userDao().activeUser.id, deprecated in version 1.19.0, will be removed in version 3.0.0")
    fun getUserID(): String {
        return userDAO.activeUser.id ?: ""
    }

    private fun clearMemory() {
        jwtToken = null
        idp = null
        languageCodes = null
        projectID = null
        gateway = null
        userId = null
        notificationChannelID = null
        isPushNotificationEnabled = null
        locationPermission = null
    }

    /**
     * Sets the gateway URL to be used for communicating with the Flybits servers.
     *
     * @param url Gateway URL to be set.
     */
    fun setGatewayURL(url: String) {
        this.gateway = url
        setStringVariable(PREF_GATEWAY_URL, url)
    }

    /**
     * Sets the IDP that was used to connect to Flybits with such as Flybits, Anonymous, OAuth, etc.
     *
     * @param idp The string representation of the IDP used to connect to Flybits with.
     */
    fun setConnectedIDP(idp: String) {
        this.idp = idp
        setStringVariable(PREF_IDP_CONNECTED, idp)
    }

    /**
     * Sets current device location permission status, it's one of always/inUse/never
     *
     * @param locationPermission Device location permission status.
     */
    fun setLocationPermission(locationPermission: String): Boolean {
        if(this.locationPermission == locationPermission){
            return false
        }
        else{
            this.locationPermission = locationPermission
            setStringVariable(PREF_LOCATION_PERMISSION, locationPermission)
            return true
        }
    }


    /**
     * Sets the Notification Channel ID that to use to display the Notification for API >=26
     *
     * @param channelID Notification Channel ID
     * @return if the value need to be updated
     */
    fun setNotificationChannel(channelID: String)  {
        setStringVariable(PREF_NOTIFICATION_CHANNEL_ID,channelID)
    }


    /**
     * Sets the if the notification get enabled
     *
     * * if OS < 26, listen to App notification value only
     * if OS>=26 and push channel ID== null, listen to App notification value only
     * if OS>=26 and push channel ID is valued, listen to App Notification value && Channel value
     * @param isPushEnabled if the push notification get enabled
     * @return if the value has been updated
     */
    fun setIsPushEnabled(isPushEnabled: Boolean) : Boolean {
        return if (this.isPushNotificationEnabled.isNullOrBlank()
            || ((this.isPushNotificationEnabled == "true") != isPushEnabled)) {
            return if(getIsPushNotificationEnabled() == isPushEnabled)
                false
            else{
                setStringVariable(PREF_IS_NOTIFICATION_ENABLED, isPushEnabled.toString())
                true
            }
        } else {
            false
        }
    }


    /**
     * Set custom Unique Device Id generator here otherwise a default one will be used
     * @param uniqueDeviceId String representing unique device ID
     */
    fun setUniqueDevice(uniqueDeviceId: String) {
        setStringVariable(PREF_UNIQUE_ID, uniqueDeviceId)
    }

    /**
     * @throws FlybitsException if no Unique Device ID stored and could not generate unique device id using uniqueDeviceIDGenerator
     *
     * @return unique device Id set by the user
     */
    fun getUniqueDeviceId() : String {
        return getStringVariable(PREF_UNIQUE_ID, "")
    }

    /**
     * Sets the unique JWT Token obtained from the Flybits Core for the user/device combination.
     *
     * @param jwtToken The unique JWT Token that is used by Flybits to identify a user/device
     * combination.
     */
    fun setJWTToken(jwtToken: String) {
        this.jwtToken = jwtToken
        setStringVariable(PREF_JWT_TOKEN, jwtToken)
    }

    /**
     * Sets the localization values of the device.
     *
     * @param listOfLanguages The array of languages that should be used for this device.
     */
    fun setLocalization(listOfLanguages: ArrayList<String>) {
        val languages = Utilities.convertLocalizationCodeToString(listOfLanguages)
        this.languageCodes = languages
        setStringVariable(PREF_LANGUAGE_CODES, languages)
    }

    /**
     * Sets the Flybits Project Identifier which can be used by other components/sdks within the
     * Flybits ecosystem.
     *
     * @param projectID The unique Flybits Project Identifier that represents this project's
     * application.
     */
    fun setProjectID(projectID: String) {
        this.projectID = projectID
        setStringVariable(PREF_PROJECT_ID, projectID)
    }


    /**
     * Sets the User Id which can be used by other components/sdks within the
     * Flybits ecosystem.
     *
     * @param userId The unique Identifier that represents current user
     */
    fun setUserId(userId: String) {
        this.userId = userId
        setStringVariable(PREF_USER_ID, userId)
    }

    abstract fun setStringVariable(key: String, value: String)

    private fun getStringVariable(key: String, default: String)
            = sharedPreferences.getString(key, default)?:""

}
