package com.flybits.commons.library.logging

import android.util.Log

/**
 * The [Displayer] class is used to display information within the logcat.
 *
 * @param tag The tag under which the log should be displayed in the logcat. The default tag is
 * "FlybitsSDK"
 * @param verbosityLevel The level of verbosity that should be displayed. Log levels enum values
 * from [VerbosityLevel]. The default is level [VerbosityLevel.NONE].
 */
class Displayer internal constructor(var tag: String = TAG,
                                     var verbosity: VerbosityLevel = VerbosityLevel.NONE) {

    companion object {
        const val TAG = "FlybitsSDK"
    }

    /**
     * Change the tag for which the information is logged under within the logcat.
     *
     * @param tag The `tag` to set for information to be displayed under.
     * @return The [Displayer] class that is used to display information within the logcat.
     */
    @Deprecated("Deprecated in version 2.1.0, will be removed in version 4.0.0",
        ReplaceWith("appendTag(String)", "com.flybits.commons.library.logging.Displayer"))
    fun changeTag(tag: String): Displayer {
        this.tag = tag
        return this
    }

    /**
     * Append [filter] to Tag for which the information is logged under within the logcat.
     *
     * @param filter The append filter on top of the `tag` to set for information to be displayed
     * under.
     * @return The [Displayer] class that is used to display information within the logcat.
     */
    fun appendTag(filter: String): Displayer {
        //If the log tag is more than 23 character it will not work. The root Tag is already 10 characters
        if (filter.length <= 13) {
            this.tag = "$TAG.$filter"
        }
        return this
    }

    /**
     * Change the verbosity level of the Displayer, which indicates what should be shown/
     *
     * @param level The level of verbosity that should be displayed. Log levels enum values
     * from [VerbosityLevel]. The default is level [VerbosityLevel.NONE].
     * @return The [Displayer] class that is used to display information within the logcat.
     */
    internal fun setVerbosity(verbosity : VerbosityLevel) : Displayer{
        this.verbosity = verbosity
        return this
    }

    /**
     * Creates a log statement in the logcat that displays information used to debug the Flybits
     * SDK.
     *
     * @param msg The message to be displayed in the logcat.
     * @return Whether or not the message was displayed based [verbosity] value.
     */
    fun d(msg: String) : Boolean {
        return if (isAboveVerbosityLevel(VerbosityLevel.DEBUG)) {
            Log.d(tag, msg)
            true
        }else {
            false
        }
    }

    /**
     * Creates a log statement in the logcat that displays an error.
     *
     * @param msg The message to be displayed in the logcat.
     * @return Whether or not the message was displayed based [verbosity] value.
     */
    fun e(msg: String) : Boolean {
        return if (isAboveVerbosityLevel(VerbosityLevel.ERROR)) {
            Log.e(tag, msg)
            true
        }else {
            false
        }
    }

    /**
     * Creates a log statement in the logcat that displays an error.
     *
     * @param msg The message to be displayed in the logcat.
     * @param exception An exception that is associated to the error
     * @return Whether or not the message was displayed based [verbosity] value.
     */
    fun e(msg: String, exception : Exception) : Boolean {
        return e("$msg - ${exception.localizedMessage}")
    }

    /**
     * Creates a log statement in the logcat that displays some information.
     *
     * @param msg The message to be displayed in the logcat.
     * @return Whether or not the message was displayed based [verbosity] value.
     */
    fun i(msg: String) : Boolean {
        return if (isAboveVerbosityLevel(VerbosityLevel.INFO)) {
            Log.i(tag, msg)
            true
        }else {
            false
        }
    }

    /**
     * Creates a log statement in the logcat that displays a warning.
     *
     * @param msg The message to be displayed in the logcat.
     * @return Whether or not the message was displayed based [verbosity] value.
     */
    fun w(msg: String) : Boolean {
        return if (isAboveVerbosityLevel(VerbosityLevel.WARNINGS)) {
            Log.w(tag, msg)
            true
        }else {
            false
        }
    }

    /**
     * Creates a log statement in the logcat that displays an exception including the
     * stacktrace.
     *
     * @param sdkMethod The class/method identifier that indicates where the exception
     * occurred.
     * @param e The exception that was thrown
     * @return Whether or not the message was displayed based [verbosity] value.
     */
    fun exception(sdkMethod : String, e: Exception) : Boolean {
        return if (isAboveVerbosityLevel(VerbosityLevel.FATAL)) {
            Log.e(tag,"There was an exception thrown for: $sdkMethod")
            e.printStackTrace()
            true
        }else {
            false
        }
    }

    private fun isAboveVerbosityLevel(checkVerbosity: VerbosityLevel) : Boolean{
        return verbosity.level >= checkVerbosity.level
    }
}
