package com.flybits.commons.library.models.internal;

import com.flybits.commons.library.deserializations.DeserializeNetworkError;
import com.flybits.commons.library.exceptions.APIUsageExceededException;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.exceptions.NotConnectedException;
import com.flybits.commons.library.http.RequestStatus;

/**
 * Defines the result of each API request made by the SDK. Included as part of this object are the
 * HTTP network status code (200, 401, 500, etc.), the response which is usually deserialized and
 * set using the {@link #setResponse(Object)} method, as well as {@link FlybitsException}s that
 * indicate why the HTTP request failed.
 *
 * @param <T> The object that is expected to be returned by the HTTP request if the request was
 *           successfully made (Produced a {@link #getStatus() of {@link RequestStatus#COMPLETED}}.
 */
public class Result<T> {

    private T response;
    private FlybitsException exception;
    private RequestStatus responseStatus;
    private String responseAsString;

    /**
     * Constructor that is used to define the status code and the body response of the HTTP request.
     *
     * @param code The HTTP status code of the HTTP result.
     * @param body The response body of the HTTP result.
     */
    public Result(int code, String body){
        setRequestStatus(code, body);
    }

    /**
     * Constructor that is used to define a result with a {@link FlybitsException} which will
     * indicate that the status of the HTTP request was {@link RequestStatus#FAILED}.
     *
     * @param exception The exception The {@link FlybitsException} that caused this request to fail.
     * @param response The response body of the HTTP result, this value may be null.
     */
    public Result(FlybitsException exception, String response){
        if (response != null) {
            this.responseAsString = response;
        }

        responseStatus          = RequestStatus.FAILED;
        this.exception          = exception;
    }

    /**
     * Get the {@link FlybitsException} that caused the HTTP request to fail. If the request has not
     * failed then this value may be null.
     *
     * @return The {@link FlybitsException} is the request has failed, null is it was successful.
     */
    public FlybitsException getException() {
        return exception;
    }

    /**
     * Get the result object expected from the HTTP request if {@link #getStatus()} is
     * {@link RequestStatus#COMPLETED}. It will return null otherwise.
     *
     * @return The generic object result that was received from the HTTP request if the
     * {@link #getStatus()} is {@link RequestStatus#COMPLETED}. Otherwise null is returned.
     */
    public T getResult() {
        return response;
    }

    /**
     * Get the string representation of the result from the HTTP request if {@link #getStatus()} is
     * {@link RequestStatus#COMPLETED}. It will return null otherwise.
     *
     * @return The string representation of the result returned from the HTTP request.
     */
    public String getResultAsString() {
        return responseAsString;
    }

    /**
     * Get the status of the HTTP request.
     *
     * @return The {@link RequestStatus} which indicates whether or not the HTTP request was
     * successful, or it failed.
     */
    public RequestStatus getStatus() {
        return responseStatus;
    }

    /**
     * Sets the response object of the HTTP network request. This should called once the HTTP result
     * has been parsed and determined that no error has occurred.
     *
     * @param response The response object that represents the result received from the HTTP network
     *                 request.
     */
    public void setResponse(T response){
        this.response   = response;
    }

    private void setRequestStatus(int code, String response){

        responseAsString  = response;

        if (code >= 200 && code < 300){

            responseStatus  = RequestStatus.COMPLETED;

        }else if (code == 401){

            responseStatus  = RequestStatus.NOT_CONNECTED;
            exception       = new NotConnectedException(response);

        }else if (code == 404){

            responseStatus  = RequestStatus.NOT_FOUND;
            exception       = new FlybitsException(response);

        }else if (code == 429){

            responseStatus  = RequestStatus.API_USAGE_EXCEEDED;
            exception       = new APIUsageExceededException("Your API quota limit has been reached. Please contact support@flybits.com to increase your quota.");

        }else{

            responseStatus  = RequestStatus.FAILED;
            exception       = new DeserializeNetworkError().fromJson(response);
            if (exception == null){
                exception   = new FlybitsException(response);
            }
        }
    }
}
