package com.flybits.commons.library.api;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.analytics.Analytics;
import com.flybits.commons.library.api.idps.FlybitsIDP;
import com.flybits.commons.library.api.idps.IDP;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ConnectionResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ConnectionResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.deserializations.DeserializeLogin;
import com.flybits.commons.library.deserializations.DeserializePagedResponse;
import com.flybits.commons.library.deserializations.DeserializeProject;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.exceptions.InvalidFlybitsManagerException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.Project;
import com.flybits.commons.library.models.User;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.commons.library.models.results.ProjectsResult;
import com.flybits.commons.library.utils.ProjectParameters;
import com.flybits.internal.db.CommonsDatabase;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import devliving.online.securedpreferencestore.DefaultRecoveryHandler;
import devliving.online.securedpreferencestore.SecuredPreferenceStore;

/**
 * The {@code FlybitsManager} class is responsible for defining all {@link FlybitsScope}s as well as
 * performing all {@link User} operations such as logging in, logging out, disabling user's account,
 * user forgotten password, and refreshing the logged in user's JWT.
 */
public class FlybitsManager {

    public static final String AUTHENTICATION_API      = "/sso/auth";
    static final String PROJECTS_API            = "/kernel/projects";
    static final String DISCONNECT_ENDPOINT     = AUTHENTICATION_API+"/logout";

    public static boolean IS_DEBUG = false;

    private Context context;
    private IDP idProvider;
    private Analytics analytics;
    private ArrayList<String> languageCodes;
    private Set<FlybitsScope> listOfScopes;
    private String projectId;

    /**
     * Constructor used to define the {@code FlybitsManager} based on the {@link Builder}
     * attributes.
     *
     * @param builder The {@link Builder} that defines all the attributes about the
     * {@code FlybitsManager}.
     */
    private FlybitsManager(Builder builder){
        listOfScopes                = builder.listOfScopes;
        context                     = builder.mContext;
        languageCodes               = builder.languageCodes;
        idProvider                  = builder.idProvider;
        projectId                   = builder.projectId;
        analytics                   = new Analytics(context);

        try{
            SecuredPreferenceStore.init(context, "flybits_con_storage", projectId, null , new DefaultRecoveryHandler());
        }catch(Exception e){
            Logger.exception("FlybitsManager.build()",e);
        }

        SharedElements.INSTANCE.migrateUnencrypted(context);

        if (languageCodes.size() == 0) {
            languageCodes.add("en");
        }

        if (projectId != null){
            SharedElements.INSTANCE.setProjectID(projectId);
        }
    }

    /**
     * The {@code Builder} class is responsible for setting all the attributes associated to the
     * {@link FlybitsManager}.
     */
    public static final class Builder {

        private final Set<FlybitsScope> listOfScopes;
        private Context mContext;
        private IDP idProvider;
        private ArrayList<String> languageCodes;
        private String projectId;

        /**
         * Constructor that initializes all the objects within the {@code Builder} class.
         *
         * @param context The {@link Context} of the application.
         */
        public Builder(@NonNull Context context) {
            this.mContext       = context;
            this.listOfScopes   = new HashSet<>();
            this.languageCodes  = new ArrayList<>();
            try {
                this.languageCodes.add(Locale.getDefault().getLanguage());
            }catch (Exception e){}
        }

        /**
         * Add a {@link FlybitsScope} to the {@link FlybitsManager}. The {@link FlybitsScope}
         * defines the SDK and all the properties associated to it.
         *
         * @param scope The {@link FlybitsScope} that is associated to this application.
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder addScope(@NonNull FlybitsScope scope) {
            listOfScopes.add(scope);
            return this;
        }

        /**
         * Build the {@link FlybitsManager} object that is used for defining the SDKs that are
         * associated to the application.
         *
         * @return {@link FlybitsManager} object which can be referenced by the SDK to get specific
         * information about the SDKs defined within the application.
         * @throws InvalidFlybitsManagerException if the {@link Builder#addScope(FlybitsScope)}
         * method was not called.
         */
        public FlybitsManager build() throws InvalidFlybitsManagerException {
            checkIfFieldsSet();
            return new FlybitsManager(this);
        }

        /**
         * Sets the {@link IDP} which should be used sign into the Flybits Account Manager.
         *
         * @param idp The {@link IDP} that should be used for registering contextual information to
         *            the account.
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setAccount(IDP idp){
            this.idProvider = idp;
            return this;
        }

        /**
         * Sets the Flybits Android commons SDK in Debug mode. While in debug mode all logs will be
         * displayed within the logcat of your Android Studio including any errors/exceptions.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setDebug(){
            IS_DEBUG    = true;
            return this;
        }

        /**
         * Sets the Language that the SDK should parse the localized values with.
         * @param languageCode The 2-letter code, such as "en", that indicates which language should
         *                     be used for parsing localized values.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setLanguage(@NonNull String languageCode){
            languageCodes.clear();
            languageCodes.add(languageCode);
            return this;
        }

        /**
         * Sets the project identifier of the application.
         *
         * @param projectId The unique GUID that represents the Project Id for the application.
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setProjectId(@NonNull String projectId){
            this.projectId = projectId;
            return this;
        }

        private void checkIfFieldsSet()throws InvalidFlybitsManagerException {

            if (listOfScopes.size() == 0){
                throw new InvalidFlybitsManagerException("You must have at least 1 scope set to create a valid FlybitsManager object");
            }

            if (languageCodes.size() != 1 && languageCodes.get(0).length() != 2){
                throw new InvalidFlybitsManagerException("Your language must be a 2-letter code. Make sure you call setLanguage(String)");
            }
        }
    }

    /**
     * Indicates that the logcat should display logs from the SDK.
     */
    public static void setDebug(){
        IS_DEBUG    = true;
    }

    /**
     * Indicates that the logcat should not display logs from the SDK.
     */
    public static void unsetDebug(){
        IS_DEBUG    = false;
    }

    /**
     * Sets the project id to another project, refreshing the current session.
     *
     * @param projectId The project id to bind to.
     * @param callback An interface to resolve successes and failures of this call.
     * @param handler The handler that the callback will be invoked through.
     * @return The {@link BasicResult} is an object that is returned from the Server after a HTTP
     * request is made.
     * @deprecated Please use {@link Project#bindProject(Context, String, BasicResultCallback)}.
     */
    @Deprecated
    public BasicResult bindProject(final String projectId, final BasicResultCallback callback, @NonNull final Handler handler) {
        return Project.bindProject(context, projectId, callback, handler);
    }

    /**
     * Sets the project id to another project, refreshing the current session.
     *
     * @param projectId The project id to bind to.
     * @param callback An interface to resolve successes and failures of this call.
     * @return The {@link BasicResult} is an object that is returned from the Server after a HTTP
     * request is made.
     * @deprecated Please use {@link Project#bindProject(Context, String, BasicResultCallback)}.
     */
    @Deprecated
    public BasicResult bindProject(final String projectId, final BasicResultCallback callback) {
        return Project.bindProject(context, projectId, callback);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param callback The response callback to run after a connection is made or fails.
     * @param handler The handler that the callback will be invoked through.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public ConnectionResult connect(final ConnectionResultCallback callback, @NonNull final Handler handler) {
        return connect(callback, true, handler);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param callback The response callback to run after a connection is made or fails.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public ConnectionResult connect(final ConnectionResultCallback callback) {
        return connect(callback, true);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param idp The {@link IDP} that should be used for registering contextual information to
     *            the account.
     * @param callback The response callback to run after a connection is made or fails.
     * @param autoUseManifestProject If true, the project id defined in the manifest will be loaded
     *                               automatically.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public ConnectionResult connect(final IDP idp, final ConnectionResultCallback callback, final boolean autoUseManifestProject) {
        idProvider  = idp;
        return connect(callback, autoUseManifestProject);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param idp The {@link IDP} that should be used for registering contextual information to the
     *            account.
     * @param callback The response callback to run after a connection is made or fails.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public ConnectionResult connect(final IDP idp, final ConnectionResultCallback callback) {
        idProvider  = idp;
        return connect(callback, true);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method. If autoUseManifestProject is true,
     * the project to load will be pulled from the manifest. Otherwise it will be in a no-project
     * state. This can only be done with the Flybits IDP.
     *
     * @param callback The response callback to run after a connection is made or fails.
     * @param autoUseManifestProject If true, the project id defined in the manifest will be loaded
     *                               automatically.
     * @param handler The handler that the callback will be invoked through.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public ConnectionResult connect(final ConnectionResultCallback callback, final boolean autoUseManifestProject, @NonNull final Handler handler){

        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ConnectionResult query = new ConnectionResult(context, callback, executorService, handler);

        executorService.execute(new Runnable() {
            @Override
            public void run() {

                try {
                    SharedElements.INSTANCE.setLocalization(languageCodes);
                    //TODO: first check if JWT is empty
                    final Result jwtResult = FlyJWT.refreshJWT(context);
                    if (jwtResult.getStatus() == RequestStatus.COMPLETED){
                        query.setResult(jwtResult);
                    }else{
                        if (idProvider != null) {

                            String body     = idProvider.getRequestBody(context, autoUseManifestProject).toString();
                            String url      = AUTHENTICATION_API + idProvider.getAuthenticationEndPoint();
                            final Result<User> authenticatedUser = FlyAway.post(context, url, body, new DeserializeLogin(), "FlybitsManager.connect", User.class);
                            if (authenticatedUser.getStatus() == RequestStatus.COMPLETED) {

                                if (idProvider instanceof FlybitsIDP){
                                    if (((FlybitsIDP) idProvider).isSendConfirmationEmail()){
                                        FlybitsIDP.sendConfirmationEmail(context, ((FlybitsIDP) idProvider).getEmail(), null);
                                    }
                                }

                                CommonsDatabase.getDatabase(context).userDao().insert(authenticatedUser.getResult());
                                SharedElements.INSTANCE.setConnectedIDP(idProvider.getProvider());

                                analytics.scheduleWorkers();

                                for (FlybitsScope scope : listOfScopes) {
                                    scope.onConnected(context, authenticatedUser.getResult());
                                }
                            }
                            query.setResult(authenticatedUser);
                        }else{
                            query.setNotConnected();
                        }
                    }
                } catch (final FlybitsException e) {
                    query.setFailed(e);
                }
            }
        });

        return query;
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method. If autoUseManifestProject is true,
     * the project to load will be pulled from the manifest. Otherwise it will be in a no-project
     * state. This can only be done with the Flybits IDP.
     *
     * @param callback The response callback to run after a connection is made or fails.
     * @param autoUseManifestProject If true, the project id defined in the manifest will be loaded
     *                               automatically.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public ConnectionResult connect(final ConnectionResultCallback callback, final boolean autoUseManifestProject){
        return connect(callback, autoUseManifestProject, new Handler(Looper.getMainLooper()));
    }

    /**
     * Clears all information associated this instance of the SDK. If you use this method
     * persistence between app launches will not be maintained, therefore it is highly recommended
     * that you use the {@link #disconnect(BasicResultCallback)} method over this {@code destroy()}
     * one.
     *
     * @param callback The callback used to indicate whether or not the HTTP request was successful.
     * @param handler The handler that the callback will be invoked through.
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult destroy(final BasicResultCallback callback, @NonNull final Handler handler){

        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(context, callback, executorService, handler);

        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    analytics.flush(null,false);
                    String jwtToken = SharedElements.INSTANCE.getSavedJWTToken();
                    final Result deleteUser = FlyAway.delete(context, User.ME_ENDPOINT, "FlybitsManager.destroy", null);
                    if (deleteUser.getStatus() == RequestStatus.COMPLETED) {

                        clearSDKData(context);
                        for (FlybitsScope scope : listOfScopes) {
                            scope.onAccountDestroyed(context, jwtToken);
                        }
                    }
                    request.setResult(deleteUser);
                } catch (final FlybitsException e) {
                    request.setFailed(e);
                }
            }
        });
        return request;
    }

    /**
     * Clears all information associated this instance of the SDK. If you use this method
     * persistence between app launches will not be maintained, therefore it is highly recommended
     * that you use the {@link #disconnect(BasicResultCallback)} method over this {@code destroy()}
     * one.
     *
     * @param callback The callback used to indicate whether or not the HTTP request was successful.
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult destroy(final BasicResultCallback callback){
        return destroy(callback, new Handler(Looper.getMainLooper()));
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context, String)}
     * method.
     *
     * @param callback The callback used to indicate whether or not the disconnection was
     * successful or not.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult disconnect(final BasicResultCallback callback){
        return disconnect(callback, false);
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context, String)} method.
     *
     * @param callback The callback used to indicate whether or not the disconnection was
     * successful or not.
     * @param disconnectOnException true if the disconnect should clear all data even if the network
     *                              request was unsuccessful, false otherwise.
     * @param handler The handler that the callback will be invoked through.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult disconnect(final BasicResultCallback callback, final boolean disconnectOnException, @NonNull final Handler handler){
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(context, callback, executorService, handler);
        executorService.execute(new Runnable() {
            @Override
            public void run() {

                String jwtToken = SharedElements.INSTANCE.getSavedJWTToken();
                try {
                    analytics.flush(null, false);

                    final Result<String> disconnected = FlyAway.post(context, DISCONNECT_ENDPOINT, "", null, "FlybitsManager.disconnect", null);
                    if (disconnected.getStatus() == RequestStatus.COMPLETED || disconnectOnException) {

                        clearSDKData(context);
                        for (FlybitsScope scope : listOfScopes) {
                            scope.onDisconnected(context, jwtToken);
                        }
                        request.setResult(new Result(200, ""));
                    }else {
                        request.setResult(disconnected);
                    }

                } catch (final FlybitsException e) {
                    if (disconnectOnException) {

                        clearSDKData(context);
                        for (FlybitsScope scope : listOfScopes) {
                            scope.onDisconnected(context, jwtToken);
                        }
                    }
                    request.setFailed(e);
                }
            }
        });
        return request;
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context, String)} method.
     *
     * @param callback The callback used to indicate whether or not the disconnection was
     * successful or not.
     * @param handler The handler that the callback will be invoked through.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult disconnect(final BasicResultCallback callback, @NonNull final Handler handler){
        return disconnect(callback, false, handler);
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context, String)} method.
     *
     * @param callback The callback used to indicate whether or not the disconnection was
     * successful or not.
     * @param disconnectOnException true if the disconnect should clear all data even if the network
     *                              request was unsuccessful, false otherwise.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult disconnect(final BasicResultCallback callback, final boolean disconnectOnException){
        return disconnect(callback, disconnectOnException, new Handler(Looper.getMainLooper()));
    }

    /**
     * Called to get all {@code Project}s that exist on this account.
     * @param params Defines paging info.
     * @param callback The callback used to collect the Project objects, or an error if one occured.
     * @param handler The handler that the callback will be invoked through.
     * @return The {@link ProjectsResult} object can be used to continue paging if more items can
     * be fetched.
     */
    public ProjectsResult getProjects(final ProjectParameters params, final PagedResultCallback<Project> callback, @NonNull final Handler handler)
    {
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ProjectsResult result = new ProjectsResult(context, params, callback, executorService, handler);

        executorService.execute(new Runnable() {
            @Override
            public void run() {

                try{
                    boolean isConnnected = !SharedElements.INSTANCE.getSavedJWTToken().equals("");
                    if (isConnnected && getIDP() instanceof FlybitsIDP){
                        DeserializeProject singleDeserializaer = new DeserializeProject();
                        DeserializePagedResponse<Project> deserializer = new DeserializePagedResponse<Project>(singleDeserializaer);
                        final Result<PagedResponse<Project>> projectList = FlyAway.get(context, PROJECTS_API, params, deserializer, "FlybitsManager.getProjects");
                        result.setResult(projectList, params);
                    }
                    else {
                        throw new FlybitsException("Either not yet connected to Flybits, or you are not using the Flybits IDP. GetProjects() only works with the Flybits IDP.");
                    }
                }catch (final FlybitsException e){
                    result.setFailed(e);
                }
            }
        });

        return result;
    }

    /**
     * Called to get all {@code Project}s that exist on this account.
     * @param params Defines paging info.
     * @param callback The callback used to collect the Project objects, or an error if one occured.
     * @return The {@link ProjectsResult} object can be used to continue paging if more items can
     * be fetched.
     */
    public ProjectsResult getProjects(final ProjectParameters params, final PagedResultCallback<Project> callback)
    {
        return getProjects(params, callback, new Handler(Looper.getMainLooper()));
    }

    /**
     * The {@code getUser} method is responsible for retrieving user information about the Flybits
     * {@link User}. The {@link User} contains basic personal attributes such as
     * {@link User#firstName} and {@link User#lastName} as well as Flybits information such as
     * {@link User#id} and {@link User#deviceID}.
     * @param context The context of the activity that is calling this method.
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     * @deprecated Please use the more optimized {@link User#getSelf} static method.
     */
    @Deprecated
    public static ObjectResult<User> getUser(final Context context){
        return User.getSelf(context, null);
    }

    /**
     * The {@code getUser} method is responsible for retrieving user information about the Flybits
     * {@link User}. The {@link User} contains basic personal attributes such as
     * {@link User#firstName} and {@link User#lastName} as well as Flybits information such as
     * {@link User#id} and {@link User#deviceID}.
     * @param context The context of the activity that is calling this method.
     * @param callback Indicates whether or not the network request was successfully made.
     * @param handler The handler that the callback will be invoked through.
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     * @deprecated Please use the more optimized {@link User#getSelf} static method.
     */
    @Deprecated
    public static ObjectResult<User> getUser(final Context context, ObjectResultCallback<User> callback, @NonNull final Handler handler){
        return User.getSelf(context, callback, handler);
    }

    /**
     * The {@code getUser} method is responsible for retrieving user information about the Flybits
     * {@link User}. The {@link User} contains basic personal attributes such as
     * {@link User#firstName} and {@link User#lastName} as well as Flybits information such as
     * {@link User#id} and {@link User#deviceID}.
     * @param context The context of the activity that is calling this method.
     * @param callback Indicates whether or not the network request was successfully made.
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     * @deprecated Please use the more optimized {@link User#getSelf} static method.
     */
    @Deprecated
    public static ObjectResult<User> getUser(final Context context, ObjectResultCallback<User> callback){
        return User.getSelf(context, callback);
    }

    /**
     * Check to see if the SDK currently has established a connection with the Flybits Server.
     *
     * @param context The Context of the application.
     * @param confirmThroughNetwork Indicates whether or not a confirm should take place with the
     *                              server. If false, this will simply check is a token is saved
     *                              locally, this token may however not be valid. If true, it will
     *                              confirm with the server that it is valid.
     * @param callback The callback that indicates whether or not the SDK is currently connected to
     *                 Flybits.
     * @param handler The handler that the callback will be invoked through.
     *
     * @return The {@link ConnectionResult} which is executing the network request (if any) to
     * determine whether or not the connection is valid.
     */
    public static ConnectionResult isConnected(@NonNull final Context context,
                                               final boolean confirmThroughNetwork,
                                               @NonNull final ConnectionResultCallback callback,
                                               @NonNull final Handler handler) {

        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        ConnectionResult query   = new ConnectionResult(context, callback, executorService, handler);
        executorService.execute(new Runnable() {
            @Override
            public void run() {

                if (confirmThroughNetwork && !SharedElements.INSTANCE.getSavedJWTToken().equals("")) {

                    try {
                        final Result getJWT = FlyJWT.refreshJWT(context);
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                if (getJWT.getStatus() == RequestStatus.NOT_CONNECTED) {
                                    callback.notConnected();
                                } else if (getJWT.getStatus() == RequestStatus.COMPLETED) {
                                    callback.onConnected();
                                } else {
                                    callback.onException(getJWT.getException());
                                }
                            }
                        });
                    } catch (final FlybitsException e) {
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                callback.onException(e);
                            }
                        });
                    }
                } else if (SharedElements.INSTANCE.getSavedJWTToken().equals("")) {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.notConnected();
                        }
                    });

                } else {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onConnected();
                        }
                    });
                }
            }
        });
        return query;
    }

    /**
     * Check to see if the SDK currently has established a connection with the Flybits Server.
     *
     * @param context The Context of the application.
     * @param confirmThroughNetwork Indicates whether or not a confirm should take place with the
     *                              server. If false, this will simply check is a token is saved
     *                              locally, this token may however not be valid. If true, it will
     *                              confirm with the server that it is valid.
     * @param callback The callback that indicates whether or not the SDK is currently connected to
     *                 Flybits.
     * @return The {@link ConnectionResult} which is executing the network request (if any) to
     * determine whether or not the connection is valid.
     */
    public static ConnectionResult isConnected(@NonNull final Context context,
                                               final boolean confirmThroughNetwork,
                                               @NonNull final ConnectionResultCallback callback){

        return isConnected(context, confirmThroughNetwork, callback, new Handler(Looper.getMainLooper()));
    }

    static void clearUserInformation(Context context) {

        SharedPreferences preferences = SharedElements.INSTANCE.getPreferences();
        SharedPreferences.Editor editor = preferences.edit();
        editor.clear();
        editor.apply();
    }

    IDP getIDP(){
        return idProvider;
    }

    Set<FlybitsScope> getScopes(){
        return listOfScopes;
    }

    private void clearSDKData(Context context){
        analytics.destroy();
        clearUserInformation(context);
        CommonsDatabase.getDatabase(context).userDao().delete();
        CommonsDatabase.getDatabase(context).cachingEntryDAO().clear();
    }
}
