package com.flybits.commons.library.http;

import android.content.Context;
import android.support.annotation.NonNull;

import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.exceptions.FlybitsDisabledException;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.exceptions.NotConnectedException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.QueryParameters;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.commons.library.utils.Utilities;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import okhttp3.FormBody;
import okhttp3.Headers;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okio.Buffer;

import static com.flybits.commons.library.api.FlyJWT.refreshJWT;

public class HttpDefaultClass<T> {
    static final String TAG_NETWORK = "Flybits_Network";
    private final int MAXTIMEOUT                    = 1;
    private int timeout;

    private Context context;
    private boolean shouldRefreshJWT;
    private OkHttpClient client;
    private Request request;

    public HttpDefaultClass(Builder builder){
        this.context            = builder.mContext;
        this.shouldRefreshJWT   = builder.mShouldRefreshJWT;
        this.request            = builder.mRequest.build();
        this.client             = HttpHelper.getClient();
        printHttpRequest();
    }

    void printHttpRequest(){

        Logger.setTag(TAG_NETWORK).d("Request URL - " + request.url().toString());

        Headers headers = request.headers();
        Set<String> headersName = headers.names();
        Iterator iterator = headersName.iterator();
        while(iterator.hasNext()){
            String iteratorAsString = (String) iterator.next();
            Logger.setTag(TAG_NETWORK).d("Request Header - (" + iteratorAsString + " = " + headers.get(iteratorAsString) + ")");
        }

        if (request.body() != null) {
            try {
                final Buffer buffer = new Buffer();
                final Request copy = request.newBuilder().build();
                copy.body().writeTo(buffer);
                String body = buffer.readUtf8();
                Logger.setTag(TAG_NETWORK).d("Request Body - " + body);
            }catch (final IOException e){

            }
        }
    }

    Request getRequest() {
        return request;
    }

    public Result<T> getResponse() throws IOException, FlybitsDisabledException {
        long timeStart      = System.currentTimeMillis();

        Response response   = client.newCall(request).execute();
        int statusCode      = response.code();

        Logger.setTag(TAG_NETWORK).d("Response - Time to Process Request: " + (System.currentTimeMillis() - timeStart));
        Logger.setTag(TAG_NETWORK).d("Response Code: " + statusCode);

        String messageBody = null;
        if (response.body() != null) {
            messageBody = response.body().string();
            Logger.setTag(TAG_NETWORK).d("Response - Result: " + messageBody);
        }

        if (statusCode == 503){
            throw new FlybitsDisabledException("Flybits is currently not available");
        }

        if (statusCode == 401 && shouldRefreshJWT){
            if (timeout < MAXTIMEOUT){
                timeout ++;
                try {
                    Result refreshResult = refreshJWT(context);
                    if (refreshResult.getStatus() == RequestStatus.COMPLETED){
                          return getResponse();
                    }else if (refreshResult.getException() instanceof NotConnectedException){
                        return new Result<T>(401, refreshResult.getException().getMessage());
                    }else{
                        return refreshResult;
                    }
                }catch (FlybitsException e){
                    Logger.exception("FlybitsHttpRequest.getResponse", e);
                    return new Result<T>(500, e.getMessage());
                }
            }
        }

        Headers responseHeaders = response.headers();
        String jwtTokenValue    = responseHeaders.get("X-Authorization");
        if (jwtTokenValue != null && jwtTokenValue.length() > 0){
            SharedElements.INSTANCE.setJWTToken(jwtTokenValue);
        }

        return new Result<T>(statusCode, messageBody);
    }

    public static class Builder {

        private Context mContext;
        private boolean mShouldRefreshJWT;
        private String mUrl;
        private Request.Builder mRequest;

        public Builder(Context context, boolean shouldRefreshJWT, String url){

            this.mContext            = context;
            this.mShouldRefreshJWT   = shouldRefreshJWT;
            this.mUrl                = url;
            setupRequest();
        }

        public Builder(Context context, boolean shouldRefreshJWT, String url, QueryParameters params){

            this.mContext            = context;
            this.mShouldRefreshJWT   = shouldRefreshJWT;

            if (params != null && params.getQueryParams() != null && params.getQueryParams().size() > 0) {
                url += "?";

                StringBuilder urlBuilder = new StringBuilder(url);
                for (Map.Entry<String, ArrayList<String>> entries : params.getQueryParams().entrySet()) {
                    {
                        for (String entry : entries.getValue()) {
                            if (entries.getKey() == null || entries.getKey().length() == 0){
                                urlBuilder.append(entry);
                            }else {
                                try {
                                    urlBuilder.append(entries.getKey()).append("=").append(URLEncoder.encode(entry, "UTF-8"));
                                }catch (UnsupportedEncodingException e){
                                    urlBuilder.append(entries.getKey()).append("=").append(entry);
                                }
                            }
                            urlBuilder.append("&");
                        }
                    }
                } url = urlBuilder.toString();
                url = url.substring(0, url.length()-1);
            }

            this.mUrl                = url;
            setupRequest();
        }

        public Builder(Context context, boolean shouldRefreshJWT, String url, Map<String, String> queryParams){

            this.mContext            = context;
            this.mShouldRefreshJWT   = shouldRefreshJWT;

            HttpUrl.Builder urlBuilder = HttpUrl
                    .parse(url)
                    .newBuilder();

            if (queryParams != null) {
                for (Map.Entry<String, String> entry : queryParams.entrySet()) {
                    urlBuilder.addQueryParameter(entry.getKey(), entry.getValue());
                }
            }

            this.mUrl                = urlBuilder.build().toString();
            setupRequest();
        }

        public Builder addHeaders(@NonNull HashMap<String, String> headers){
            if (headers != null && headers.size() > 0) {
                headers(headers);
            }
            return this;
        }

        public Builder addHeader(String key, String value){
            HashMap<String, String> listOfHeaders   = new HashMap<>();
            listOfHeaders.put(key, value);
            headers(listOfHeaders);
            return this;
        }

        public HttpDefaultClass delete(){
            mRequest.delete();
            return new HttpDefaultClass(this);
        }

        public HttpDefaultClass delete(String json){
            RequestBody requestBody    = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), json);
            mRequest.delete(requestBody);
            return new HttpDefaultClass(this);
        }

        public HttpDefaultClass get(){
            mRequest.get();
            return new HttpDefaultClass(this);
        }

        public HttpDefaultClass post(RequestBody requestBody){
            mRequest.post(requestBody);
            return new HttpDefaultClass(this);
        }

        public HttpDefaultClass post(String json){
            RequestBody requestBody     = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), json);
            mRequest.post(requestBody);
            return new HttpDefaultClass(this);
        }

        public HttpDefaultClass post(HashMap<String, String> formParams){

            FormBody.Builder formBody = new FormBody.Builder();
            Iterator it = formParams.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<String, String> pair = (Map.Entry)it.next();
                formBody.add(pair.getKey(), pair.getValue());
                it.remove();
            }
            mRequest.post(formBody.build());
            return new HttpDefaultClass(this);
        }

        public HttpDefaultClass put(RequestBody requestBody){
            mRequest.put(requestBody);
            return new HttpDefaultClass(this);
        }

        public HttpDefaultClass put(String json){
            RequestBody requestBody     = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), json);
            mRequest.put(requestBody);
            return new HttpDefaultClass(this);
        }

        private void setupRequest(){
            mRequest = new Request.Builder()
                    .url(mUrl)
                    .header("X-User-Agent", Utilities.getUserAgentAsJSON(mContext))
                    .header("Accept", "application/json")
                    .header("Content-Type", "application/json");

            if (SharedElements.INSTANCE.getEnabledLanguagesAsString().length() > 0){
                mRequest.header("Accept-Language", SharedElements.INSTANCE.getEnabledLanguagesAsString());
            }

            String jwtToken = SharedElements.INSTANCE.getSavedJWTToken();
            if (jwtToken != null && jwtToken.length() > 0){
                mRequest.header("X-Authorization", SharedElements.INSTANCE.getSavedJWTToken() );
            }
        }

        private void headers(HashMap<String, String> headers){
            if (headers != null){
                for (Map.Entry<String, String> entry : headers.entrySet()){
                    mRequest.header(entry.getKey(), entry.getValue());
                }
            }
        }
    }
}
