package com.flybits.commons.library.models;

import android.util.Base64;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * This class contains the various deserialized payload data parsed from a JWT token sent by the
 * Flybits servers.
 */
public class Jwt {

    private long exp;
    private long nbf;
    private String userID;
    private String deviceID;
    private String tenantID;
    private boolean isServiceAccount;

    private Jwt(String userID, String deviceID, String tenantID, long exp, long nbf, boolean isServiceAccount)
    {
        this.userID = userID;
        this.deviceID = deviceID;
        this.tenantID = tenantID;
        this.exp = exp;
        this.nbf = nbf;
        this.isServiceAccount = isServiceAccount;
    }

    /**
     * Decodes a JWT token into a {@link Jwt} object.
     * @param token The token to deserialize.
     * @return An object containing the payload data.
     */
    public static Jwt decodeJWTToken(String token)
    {
        if (!token.contains("."))
            throw new IllegalArgumentException("Not a valid JWT token.");

        String[] split = token.split("\\.");

        if (split.length < 2)
            throw new IllegalArgumentException("JWT token missing payload.");

        byte[] data = Base64.decode(split[1], Base64.DEFAULT);
        String decoded = new String(data);

        try {
            JSONObject object = new JSONObject(decoded);

            String userId = object.getString("userID");
            String deviceId = object.getString("deviceID");
            String projectId = object.getString("tenantID");
            long expires = object.getLong("exp");
            long nbf = object.getLong("nbf");
            //boolean isServiceAccount = object.getBoolean("isServiceAccount");

            return new Jwt(userId, deviceId, projectId, expires, nbf, false);
        } catch (JSONException e) {
            throw new IllegalArgumentException("JWT token payload invalid.");
        }
    }

    /**
     * Returns the user id in this token.,
     * @return The user id.
     */
    public String getUserID()
    {
        return userID;
    }

    /**
     * Returns the device id in this token.
     * @return The device id.
     */
    public String getDeviceID()
    {
        return deviceID;
    }

    /**
     * Returns the project id in this token.,
     * @return The project id.
     */
    public String getProjectID()
    {
        return tenantID;
    }

    /**
     * Returns true if a project is bound to this token or not.
     * @return True if the project is bound.
     */
    public boolean isProjectBound()
    {
        return !tenantID.equals("00000000-0000-0000-0000-000000000000");
    }

    /**
     * Returns the user id in this token.,
     * @return The user id.
     */
    /*public boolean isServiceAccount()
    {
        return isServiceAccount;
    }*/

    /**
     * Returns when this token will expire.
     * @return The token expiry time.
     */
    public long getExpiry()
    {
        return exp;
    }
}
