package com.flybits.commons.library.analytics

import android.content.Context
import androidx.work.*
import com.flybits.commons.library.api.FlyAway
import com.flybits.commons.library.http.RequestStatus
import com.flybits.commons.library.models.CtxData
import com.flybits.internal.db.CommonsDatabase
import org.json.JSONObject
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit

/**
 * This class is responsible for tracking analytics data, this including: scheduling workers
 * , storing the data in the local database, and formatting the data in the expected format.
 *
 */
open class Analytics(private val context: Context) {

    /**
     * Represents an application event that will be tracked by the Flybits Analytics system.
     *
     * @param analyticsScope Scope of the analytics e.x push or content.
     * @param action Action that occurred e.x viewed or engaged.
     * @param dataId Identification of the data associated with the event.
     * @param timestamp When the event occurred.
     *
     */
    data class Event(val analyticsScope: String, val action: String, val dataId: String, val timestamp: Long)

    private val ctxDataDAO = CommonsDatabase.getDatabase(context).ctxDataDAO()
    private val workManager = WorkManager.getInstance()

    companion object {
        private const val UPLOAD_INTERVAL_METERED = 60L
        private const val UPLOAD_INTERVAL_UNMETERED = 20L
        private const val CTX_DATA_ENDPOINT = "/context/ctxdata"
        const val TAG_METERED_WORK = "com.flybits.commons.library.analytics.meteredWork"
        const val TAG_UNMETERED_WORK = "com.flybits.commons.library.analytics.unmeteredWork"
    }

    private fun Event.toCtxData(): CtxData{
        val analyticsJson = JSONObject()
        analyticsJson.put("dataTypeID","ctx.flybits.$analyticsScope")
        val valueJson = JSONObject()
        valueJson.put("query.$action.$dataId", true)
        analyticsJson.put("value", valueJson)
        return CtxData(timestamp = timestamp, value = analyticsJson.toString())
    }

    /**
     * Flush all data from the ctxDatabase to the server then delete the data if it was
     * sent with success.
     *
     * @param callback Callback which will be informed about whether the flush succeeded or failed.
     * @param async Whether or not to execute on a separate thread, true by default.
     *
     */
    fun flush( callback: ((success: Boolean, exception: Exception?) -> Unit)? = null, async: Boolean = true){
        if (async) Executors.newSingleThreadExecutor().execute {
            flush(callback)
        }else flush(callback)
    }

    private fun flush(callback: ((success: Boolean, exception: Exception?) -> Unit)? = null){
        val ctxDataDAO = CommonsDatabase.getDatabase(context).ctxDataDAO()
        val ctxData = ctxDataDAO.getAll()

        if (!ctxData.isEmpty()) {
            val body = ctxData.joinToString(separator = ",", prefix = "[", postfix = "]") { it.value }
            val result = FlyAway.post<Any>(context
                    , CTX_DATA_ENDPOINT, body, null
                    , "AnalyticsWorker.doWork", null)

            if (result.status == RequestStatus.COMPLETED) {
                //We cannot just delete all since by the time the response comes back an item which has not been sent may have been added
                ctxDataDAO.deleteMany(ctxData)
                callback?.invoke(true, null)
            }else{
                callback?.invoke(false, result.exception)
            }
        }else{
            callback?.invoke(true, null)
        }
    }

    /**
     * Enqueue a {@link PeriodicWorkRequest} which involves a worker periodically
     * uploading data from the local database to the server pertaining to analytics.
     * Data will be sent to the server for both a metered and unmetered connection.
     * Metered connection data will be sent less frequently.
     *
     * @return false if both metered and unmetered workers have been scheduled, true otherwise.
     */
    fun scheduleWorkers(){
        scheduleMeteredWorker()
        scheduleUnmeteredWorker()
    }

    fun cancelWorkers(){
        cancelUnmeteredWork()
        cancelMeteredWork()
    }

    /**
     * Destroy all stored analytics data, and workers responsible for uploading it.
     *
     */
    fun destroy(){
        cancelWorkers()
        Executors.newSingleThreadExecutor().execute{
            ctxDataDAO.deleteAll()
        }
    }

    private fun cancelUnmeteredWork(){
        workManager.cancelAllWorkByTag(TAG_UNMETERED_WORK)

    }

    private fun cancelMeteredWork(){
        workManager.cancelAllWorkByTag(TAG_METERED_WORK)

    }

    private fun scheduleMeteredWorker(){
        val constraintsMetered = Constraints.Builder()
                .setRequiredNetworkType(NetworkType.METERED)
                .build()

        val workRequestMetered = PeriodicWorkRequest
                .Builder(AnalyticsWorker::class.java, UPLOAD_INTERVAL_METERED, TimeUnit.MINUTES)
                .setConstraints(constraintsMetered)
                .addTag(TAG_METERED_WORK)
                .build()

        workManager.enqueue(workRequestMetered)
    }

    private fun scheduleUnmeteredWorker(){
        val constraintsMetered = Constraints.Builder()
                .setRequiredNetworkType(NetworkType.UNMETERED)
                .build()

        val workRequestMetered = PeriodicWorkRequest
                .Builder(AnalyticsWorker::class.java, UPLOAD_INTERVAL_UNMETERED, TimeUnit.MINUTES)
                .setConstraints(constraintsMetered)
                .addTag(TAG_UNMETERED_WORK)
                .build()

        workManager.enqueue(workRequestMetered)
    }

    /**
     * Queue an event for tracking which will be uploaded to the server in the future.
     *
     * Runs on a separate thread.
     *
     * @param event Analytics event that you want to track.
     */
    protected fun track(event: Event){
        Executors.newSingleThreadExecutor().execute{
            ctxDataDAO.insert(event.toCtxData())
        }
    }

}