package com.easefun.polyvsdk.sub.screenshot;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.support.annotation.Nullable;

import com.easefun.polyvsdk.sub.auxilliary.CheckThrows;
import com.easefun.polyvsdk.sub.auxilliary.IOUtil;
import com.easefun.polyvsdk.sub.auxilliary.MediaData;
import com.easefun.polyvsdk.sub.auxilliary.RetrofitHelper;
import com.easefun.polyvsdk.sub.auxilliary.SDCardUtil;
import com.easefun.polyvsdk.sub.auxilliary.cache.auxiliary.Md5Util;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;

/**
 * 网络截图
 */
public class PolyvScreenShot {
    private static final String SNAPSHOT_STR = "polyvsnapshot";
    private Context context;

    public PolyvScreenShot(Context context) {
        this.context = context.getApplicationContext();
        RetrofitHelper.init(context);
    }

    public class ScreenshotCallBack implements Callback<ResponseBody> {

        private boolean update;
        private String filePath;
        private ScreenshotListener listener;

        public ScreenshotCallBack(boolean update, String filePath, ScreenshotListener listener) {
            this.update = update;
            this.filePath = filePath;
            this.listener = listener;
        }

        @Override
        public void onResponse(Call<ResponseBody> call, Response<ResponseBody> response) {
            int code = response.code();
            if (code != 200 && code != 206) {
                onFailure(call, new Exception("response code is " + code));
                return;
            }
            FileOutputStream out = null;
            try {
                byte[] datas = response.body().bytes();
                Bitmap bitmap = BitmapFactory.decodeByteArray(datas, 0, datas.length);
                if (bitmap == null) {
                    onFailure(call, new Exception("decode data fail, bitmap is null"));
                    return;
                }
                out = new FileOutputStream(filePath);
                boolean result = bitmap.compress(Bitmap.CompressFormat.JPEG, 100, out);
                if (!result) {
                    onFailure(call, new Exception("bitmap compress fail"));
                    return;
                }
                if (listener != null)
                    listener.success(filePath);
                if (update)
                    MediaData.updateToMediaDatabase(context, filePath);
            } catch (FileNotFoundException e) {
                onFailure(call, e);
            } catch (IOException e) {
                onFailure(call, e);
            } finally {
                IOUtil.closeIO(out);
            }
        }

        @Override
        public void onFailure(Call<ResponseBody> call, Throwable t) {
            if (listener != null)
                listener.fail(t);
        }
    }

    public interface ScreenshotListener {
        void fail(Throwable t);

        void success(String filePath);
    }

    public Call<ResponseBody> snapshot(String vid, int bitrate, int currenttime_second, ScreenshotListener listener) {
        return snapshot(vid, bitrate, currenttime_second, listener, SDCardUtil.createPath(context, SNAPSHOT_STR));
    }

    public Call<ResponseBody> snapshot(String vid, int bitrate, int currenttime_second, ScreenshotListener listener, String savePath) {
        return snapshot(vid, bitrate, currenttime_second, listener, savePath, false);
    }

    /**
     * 网络截图
     * @param vid 视频id
     * @param bitrate 码率
     * @param currenttime_second 当前播放的时间，单位：秒
     * @param listener 监听器
     * @param savePath 保存路径
     * @param updateToMediaDatabase 是否立刻更新到媒体数据库中
     * @return 请求对象，可能为null
     */
    public @Nullable Call<ResponseBody> snapshot(String vid, int bitrate, int currenttime_second, ScreenshotListener listener, String savePath, boolean updateToMediaDatabase) {
        try {
            return snapshot_t(vid, bitrate, currenttime_second, listener, savePath, updateToMediaDatabase);
        } catch (Exception e) {
            if (listener != null)
                listener.fail(e);
        }
        return null;
    }

    public Call<ResponseBody> snapshot_t(String vid, int bitrate, int currenttime_second, ScreenshotListener listener, String savePath, boolean updateToMediaDatabase) throws Exception {
        CheckThrows.checkNotNull("vid", vid);
        CheckThrows.checkRange_lessequal0("bitrate", bitrate);
        CheckThrows.checkRange_less0("currenttime_second", currenttime_second);
        CheckThrows.checkNotNull("savePath", savePath);
        CheckThrows.checkNotFile("savePath", savePath);
        Map<String, Object> map = new HashMap<>();
        map.put("vid", vid);
        map.put("level", bitrate);
        map.put("second", currenttime_second);
        map.put("sign", Md5Util.getMd5(vid + bitrate + currenttime_second + SNAPSHOT_STR));
        Call<ResponseBody> screenshotCall = RetrofitHelper.getPolyvGoApi()
                .screenshot(map);
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append(vid).append("_").append(new SimpleDateFormat("yyyy-MM-dd_kk:mm:ss").format(new Date())).append(".jpg");
        String filePath = new File(savePath, stringBuilder.toString()).getAbsolutePath();
        screenshotCall.enqueue(new ScreenshotCallBack(updateToMediaDatabase, filePath, listener));
        return screenshotCall;
    }
}