package com.easefun.polyvsdk.sub.danmaku.main;

import android.content.Context;
import android.support.annotation.IntRange;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.easefun.polyvsdk.sub.danmaku.auxiliary.BiliDanmukuParser;
import com.easefun.polyvsdk.sub.danmaku.auxiliary.BilibiliDanmakuTransfer;
import com.easefun.polyvsdk.sub.auxilliary.CheckThrows;
import com.easefun.polyvsdk.sub.auxilliary.LogUtil;
import com.easefun.polyvsdk.sub.auxilliary.RetrofitHelper;
import com.easefun.polyvsdk.sub.danmaku.entity.PolyvDanmakuEntity;
import com.easefun.polyvsdk.sub.danmaku.entity.PolyvDanmakuInfo;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import master.flame.danmaku.danmaku.loader.ILoader;
import master.flame.danmaku.danmaku.loader.IllegalDataException;
import master.flame.danmaku.danmaku.loader.android.DanmakuLoaderFactory;
import master.flame.danmaku.danmaku.model.android.Danmakus;
import master.flame.danmaku.danmaku.parser.BaseDanmakuParser;
import master.flame.danmaku.danmaku.parser.IDataSource;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;

/**
 * 弹幕管理类
 */
public class PolyvDanmakuManager {

    public PolyvDanmakuManager(Context context) {
        RetrofitHelper.init(context);
    }

    public static class GetDanmakuCallBack implements Callback<List<PolyvDanmakuInfo>> {
        GetDanmakuListener getDanmakuListener;

        public GetDanmakuCallBack(GetDanmakuListener getDanmakuListener) {
            this.getDanmakuListener = getDanmakuListener;
        }

        @Override
        public void onResponse(Call<List<PolyvDanmakuInfo>> call, Response<List<PolyvDanmakuInfo>> response) {
            int code = response.code();
            if (code != 200 && code != 206) {
                onFailure(call, new Exception("response code is " + code));
                return;
            }
            List<PolyvDanmakuInfo> ls_danmakuInfos = response.body();
            PolyvDanmakuEntity danmakuEntity = new PolyvDanmakuEntity();
            danmakuEntity.setAllDanmaku(ls_danmakuInfos);
            try {
                if (getDanmakuListener != null)
                    getDanmakuListener.success(createParser(BilibiliDanmakuTransfer.transferToInputStream(ls_danmakuInfos)), danmakuEntity);
            } catch (IOException e) {
                onFailure(call, e);
            }
        }

        @Override
        public void onFailure(Call<List<PolyvDanmakuInfo>> call, Throwable t) {
            if (getDanmakuListener != null)
                getDanmakuListener.fail(t);
        }
    }

    public static class SendDanmakuCallBack implements Callback<ResponseBody> {
        SendDanmakuListener sendDanmakuListener;

        public SendDanmakuCallBack(SendDanmakuListener sendDanmakuListener) {
            this.sendDanmakuListener = sendDanmakuListener;
        }

        @Override
        public void onResponse(Call<ResponseBody> call, Response<ResponseBody> response) {
            int code = response.code();
            if (code != 200 && code != 206) {
                onFailure(call, new Exception("response code is " + code));
                return;
            }
            try {
                if (sendDanmakuListener != null)
                    sendDanmakuListener.success(response.body().string());
            } catch (IOException e) {
                onFailure(call, e);
            }
        }

        @Override
        public void onFailure(Call<ResponseBody> call, Throwable t) {
            if (sendDanmakuListener != null)
                sendDanmakuListener.fail(t);
        }
    }

    public interface GetDanmakuListener {
        void fail(Throwable t);

        void success(BaseDanmakuParser danmakuParser, PolyvDanmakuEntity danmakuEntity);
    }

    public interface SendDanmakuListener {
        void fail(Throwable t);

        void success(String responseMsg);
    }

    public Call<ResponseBody> sendDanmaku(@NonNull PolyvDanmakuInfo danmakuInfo, SendDanmakuListener sendDanmakuListener) {
        try {
            return sendDanmaku_t(danmakuInfo, sendDanmakuListener);
        } catch (Exception e) {
            if (sendDanmakuListener != null)
                sendDanmakuListener.fail(e);
        }
        return null;
    }

    /**
     * 发送弹幕
     *
     * @param danmakuInfo         弹幕实体对象
     * @param sendDanmakuListener 监听器
     * @return 请求对象，可能为null
     */
    public @Nullable Call<ResponseBody> sendDanmaku_t(@NonNull PolyvDanmakuInfo danmakuInfo, SendDanmakuListener sendDanmakuListener) throws Exception {
        CheckThrows.checkNotNull("danmakuInfo", danmakuInfo);
        CheckThrows.checkNotNull("danmakuInfo.vid", danmakuInfo.vid);
        Map<String, Object> map = new HashMap<>();
        map.put("vid", danmakuInfo.vid);
        map.put("msg", danmakuInfo.getMsg());
        map.put("time", danmakuInfo.getTime());
        map.put("fontSize", danmakuInfo.getFontSize());
        map.put("fontMode", danmakuInfo.getFontMode());
        map.put("fontColor,", danmakuInfo.getFontColor());
        Call<ResponseBody> sendCall = RetrofitHelper.getPolyvGoApi()
                .sendDanmaku(map);
        sendCall.enqueue(new SendDanmakuCallBack(sendDanmakuListener));
        return sendCall;
    }

    /**
     * 获取所有弹幕
     *
     * @param vid
     */
    public Call<List<PolyvDanmakuInfo>> getAllDanmaku(@NonNull String vid, GetDanmakuListener listener) {
        return getDanmaku(vid, 0, listener);
    }

    /**
     * 获取弹幕
     * @param vid
     * @param limit 拿取的数目，<=0时获取所有
     * @param listener
     * @return 请求对象，可能为null
     */
    public @Nullable Call<List<PolyvDanmakuInfo>> getDanmaku(@NonNull String vid, @IntRange(from = 1) int limit, GetDanmakuListener listener) {
        try {
            return getDanmaku_t(vid, limit, listener);
        } catch (Exception e) {
            if (listener != null)
                listener.fail(e);
        }
        return null;
    }

    /**
     * 获取弹幕
     *
     * @param vid
     * @param limit
     */
    public Call<List<PolyvDanmakuInfo>> getDanmaku_t(@NonNull String vid, @IntRange(from = 1) int limit, GetDanmakuListener listener) throws Exception {
        CheckThrows.checkNotNull("vid", vid);
        Map<String, Object> map = new HashMap<>();
        map.put("vid", vid);
        if (limit > 0)
            map.put("limit", limit);
        Call<List<PolyvDanmakuInfo>> getCall = RetrofitHelper.getPolyvGoApi()
                .getDanmaku(map);
        getCall.enqueue(new GetDanmakuCallBack(listener));
        return getCall;
    }

    /**
     * 获取弹幕解析器
     *
     * @param stream
     * @return
     */
    private static BaseDanmakuParser createParser(InputStream stream) {
        if (stream == null) {
            return new BaseDanmakuParser() {
                @Override
                protected Danmakus parse() {
                    return new Danmakus();
                }
            };
        }
        ILoader loader = DanmakuLoaderFactory.create(DanmakuLoaderFactory.TAG_BILI);
        try {
            loader.load(stream);
        } catch (IllegalDataException e) {
            LogUtil.w("createParser", e);
        }
        BaseDanmakuParser parser = new BiliDanmukuParser();
        IDataSource<?> dataSource = loader.getDataSource();
        parser.load(dataSource);
        return parser;
    }

    /**
     * 获取弹幕解析器
     *
     * @param entity
     * @return
     */
    public static BaseDanmakuParser createParser(PolyvDanmakuEntity entity) {
        try {
            return createParser(BilibiliDanmakuTransfer.transferToInputStream(entity.getAllDanmaku()));
        } catch (IOException e) {
            return new BaseDanmakuParser() {
                @Override
                protected Danmakus parse() {
                    return new Danmakus();
                }
            };
        }
    }
}
