package com.easefun.polyvsdk.sub.auxilliary.cache.image;

import android.annotation.TargetApi;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.support.annotation.NonNull;
import android.support.v4.util.LruCache;
import android.text.style.ImageSpan;

import com.easefun.polyvsdk.sub.auxilliary.IOUtil;
import com.easefun.polyvsdk.sub.auxilliary.LogUtil;
import com.easefun.polyvsdk.sub.auxilliary.cache.DiskLruCache;
import com.easefun.polyvsdk.sub.auxilliary.cache.auxiliary.BitmapUtil;
import com.easefun.polyvsdk.sub.auxilliary.cache.auxiliary.Md5Util;
import com.easefun.polyvsdk.sub.auxilliary.SDCardUtil;
import com.easefun.polyvsdk.sub.auxilliary.cache.auxiliary.AppHelper;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;

import pl.droidsonroids.gif.GifDrawable;
import pl.droidsonroids.gif.GifImageSpan;
import pl.droidsonroids.gif.RelativeImageSpan;


public class ImageLoader {
    private static ImageLoader imageLoader;
    private Handler mUIHandler;
    private List<String> urls;
    private Map<ImageLoaderListener, String> listeners;

    private LruCache<String, ImageData> mLruCache;
    private DiskLruCache diskCache;
    private static final int CACHE_MEMORY = (int) (Runtime.getRuntime().maxMemory() / 8);
    private static final int DISK_CACHE_DEFAULT_SIZE = 20 * 1024 * 1024;
    private static final String DISK_CACHE_PATH = "vlms";
    private Thread mPoolThread;
    private Handler mPoolThreadHandler;
    private ExecutorService mThreadPool;
    private int mThreadCount;
    private Type mType = Type.LIFO;
    private LinkedList<Runnable> mTaskQueue;

    private Semaphore mSemaphorePoolThreadHandler = new Semaphore(0);
    private Semaphore mSemaphoreThreadPool;

    public enum Type {
        FIFO, LIFO;
    }

    private ImageLoader(Context context, int threadCount, Type type, String cachePath, long fileCacheSize, long memoryCacheSize) {
        type = type == null ? Type.LIFO : type;
        init(context, threadCount, type, cachePath, fileCacheSize, memoryCacheSize);
        urls = new ArrayList<>();
        listeners = new HashMap<>();
        mUIHandler = new Handler() {
            @Override
            public void handleMessage(Message msg) {
                HandlerData data = (HandlerData) msg.obj;
                if (data.t != null) {
                    data.listener.fail(data.t);
                } else if (data.imageData != null) {
                    data.listener.success(data.imageData, data.urlStr);
                } else {
                    data.listener.fail(new Exception("imagedata is null"));
                }
            }
        };
    }

    public static ImageLoader getInstance(Context context, int threadCount, Type type) {
        return getInstance(context, threadCount, type, DISK_CACHE_PATH, DISK_CACHE_DEFAULT_SIZE, CACHE_MEMORY);
    }

    public static ImageLoader getInstance(Context context, int threadCount, Type type, String cachePath, long fileCacheSize, long memoryCacheSize) {
        if (imageLoader == null) {
            synchronized (ImageLoader.class) {
                if (imageLoader == null) {
                    imageLoader = new ImageLoader(context, threadCount, type, cachePath, fileCacheSize, memoryCacheSize);
                }
            }
        }
        return imageLoader;
    }

    /**
     * 初始化
     *
     * @param threadCount
     * @param type
     */
    private void init(Context context, int threadCount, Type type, String filePath, long fileCacheSize, long memoryCacheSize) {
        initBackThread();

        // 获取我们应用的最大可用内存
        mLruCache = new LruCache<String, ImageData>((int) memoryCacheSize) {
            @Override
            protected int sizeOf(String key, ImageData imageData) {
                return (int) imageData.getLength();
            }
        };

        try {
            File cacheDir = SDCardUtil.getDiskCacheDir(context, filePath);
            if (!cacheDir.exists())
                cacheDir.mkdirs();
            diskCache = DiskLruCache.open(cacheDir, AppHelper.getVersionCode(context), 1, fileCacheSize);
        } catch (IOException e) {
            LogUtil.w("init", e);
        }

        // 创建线程池
        mThreadPool = Executors.newFixedThreadPool(threadCount);
        mTaskQueue = new LinkedList<Runnable>();
        mType = type;
        mThreadCount = threadCount;
        mSemaphoreThreadPool = new Semaphore(threadCount);
    }

    private class HandlerData {
        @NonNull
        ImageLoaderListener listener;
        Throwable t;
        ImageData imageData;
        String urlStr;
    }

    @Deprecated
    public LruCache<String, ImageData> getLruCache() {
        return mLruCache;
    }

    @Deprecated
    public DiskLruCache getDiskCache() {
        return diskCache;
    }

    @Deprecated
    public ExecutorService getExecutorService() {
        return mThreadPool;
    }

    private void initBackThread() {
        mPoolThread = new Thread() {
            @Override
            public void run() {
                Looper.prepare();
                mPoolThreadHandler = new Handler() {
                    @Override
                    public void handleMessage(Message msg) {
                        // 线程池去取出一个任务进行执行
                        if (!mThreadPool.isShutdown())
                            mThreadPool.execute(getTask());
                        try {
                            // 轮询线程的信号量--(不再接收新的信息)
                            mSemaphoreThreadPool.acquire();
                        } catch (InterruptedException e) {
                        }
                    }
                };
                // 释放一个信号量
                mSemaphorePoolThreadHandler.release();
                Looper.loop();
            }
        };

        mPoolThread.start();
    }

    public void release() {
        shutdownNow();
        clearLruCache();
    }

    public void shutdownNow() {
        mThreadPool.shutdownNow();
        mTaskQueue.clear();
        urls.clear();
        listeners.clear();
        mThreadPool = Executors.newFixedThreadPool(mThreadCount);
        mSemaphoreThreadPool.release(mThreadCount);
    }

    public void clearLruCache() {
        mLruCache.evictAll();
    }

    private synchronized Runnable getTask() {
        if (mTaskQueue.size() > 0) {
            if (mType == Type.FIFO) {
                return mTaskQueue.removeFirst();
            } else if (mType == Type.LIFO) {
                return mTaskQueue.removeLast();
            }
        }
        return null;
    }

    private synchronized void addTask(Runnable runnable) {
        if (runnable == null)
            return;
        mTaskQueue.add(runnable);
        // if(mPoolThreadHandler==null)wait();
        try {
            if (mPoolThreadHandler == null)
                mSemaphorePoolThreadHandler.acquire();
        } catch (InterruptedException e) {
        }
        mPoolThreadHandler.sendEmptyMessage(0x123);
    }

    private ImageData getBitmapFromLruCache(String urlStr) {
        return mLruCache.get(urlStr);
    }

    @TargetApi(12)
    private void addBitmapToLruCache(String urlStr, ImageData imageData) {
        if (getBitmapFromLruCache(urlStr) == null) {
            if (imageData != null) {
                mLruCache.put(urlStr, imageData);
            }
        }
    }

    public interface ImageLoaderListener {
        void fail(Throwable t);

        void success(ImageData imageData, String urlStr);
    }

    public ImageData start(@NonNull Context context, @NonNull int[] wh, @NonNull String urlStr, @NonNull ImageLoaderListener listener) {
//        CheckRuntime.checkNotNull(context, "context");
//        CheckRuntime.checkNotNull(urlStr, "urlStr");
//        CheckRuntime.checkNotNull(listener, "listener");
        ImageData imageData = getBitmapFromLruCache(urlStr);
        if (imageData != null) {
//            successHandler(imageData, urlStr, listener);
        } else {
            addTask(buildTalk(context, wh, urlStr, listener));
        }
        return imageData;
    }

    private boolean initTask(String urlStr, ImageLoaderListener listener) {
        synchronized (ImageLoader.class) {
            if (urls.contains(urlStr)) {
                listeners.put(listener, urlStr);
                return false;
            } else {
                urls.add(urlStr);
                return true;
            }
        }
    }

    private void successHandler(ImageData imageData, String urlStr, ImageLoaderListener listener) {
        if (listener != null) {
            HandlerData handlerData = new HandlerData();
            handlerData.listener = listener;
            handlerData.imageData = imageData;
            handlerData.urlStr = urlStr;
            Message message = mUIHandler.obtainMessage();
            message.obj = handlerData;
            mUIHandler.sendMessage(message);
        }
    }

    private void failHandler(Throwable t, String urlStr, ImageLoaderListener listener) {
        if (listener != null) {
            HandlerData handlerData = new HandlerData();
            handlerData.listener = listener;
            handlerData.t = t;
            handlerData.urlStr = urlStr;
            Message message = mUIHandler.obtainMessage();
            message.obj = handlerData;
            mUIHandler.sendMessage(message);
        }
    }

    private void finishTask(ImageData imageData, String urlStr, ImageLoaderListener listener) {
        synchronized (ImageLoader.class) {
            urls.remove(urlStr);
            successHandler(imageData, urlStr, listener);
            addBitmapToLruCache(urlStr, imageData);
            if (listeners.size() > 0) {
                List<ImageLoaderListener> removeListeners = new ArrayList<>();
                for (ImageLoaderListener waitListener : listeners.keySet()) {
                    String waitUrl = listeners.get(waitListener);
                    if (urlStr.equals(waitUrl)) {
                        successHandler(imageData, urlStr, waitListener);
                        removeListeners.add(waitListener);
                    }
                }
                for (ImageLoaderListener removeListener : removeListeners)
                    listeners.remove(removeListener);
            }
            //释放信号给下一个任务
            mSemaphoreThreadPool.release();
        }
    }

    private void failTask(Throwable t, String urlStr, ImageLoaderListener listener) {
        synchronized (ImageLoader.class) {
            urls.remove(urlStr);
            failHandler(t, urlStr, listener);
            if (listeners.size() > 0) {
                List<ImageLoaderListener> removeListeners = new ArrayList<>();
                for (ImageLoaderListener waitListener : listeners.keySet()) {
                    String waitUrl = listeners.get(waitListener);
                    if (urlStr.equals(waitUrl)) {
                        failHandler(t, urlStr, listener);
                        removeListeners.add(waitListener);
                    }
                }
                for (ImageLoaderListener removeListener : removeListeners)
                    listeners.remove(removeListener);
            }
            //释放信号给下一个任务
            mSemaphoreThreadPool.release();
        }
    }

    private Runnable buildTalk(final Context context, final int[] wh, final String urlStr, final ImageLoaderListener listener) {
        if (!initTask(urlStr, listener))
            return null;
        return new Runnable() {
            @Override
            public void run() {
                String key = Md5Util.hashKeyForDisk(urlStr);
                try {
                    DiskLruCache.Snapshot snapShot = diskCache.get(key);
                    if (snapShot != null) {
                        finishTask(generateImageData(context, urlStr, wh), urlStr, listener);
                        return;
                    }
                } catch (IOException e) {
                    LogUtil.w("buildTalk", e);
                }
                // 下载成功后直接将图片流写入文件缓存
                try {
                    DiskLruCache.Editor editor = diskCache.edit(key);
                    if (editor != null) {
                        if (downloadUrlToStream(urlStr, editor) && !diskCache.isClosed())
                            editor.commit();
                        else if (!diskCache.isClosed()) {
                            editor.abort();
                            diskCache.flush();
                            failTask(new Exception("downloadUrlToStream is fail"), urlStr, listener);
                            return;
                        }
                        if (!diskCache.isClosed())
                            diskCache.flush();
                        finishTask(generateImageData(context, urlStr, wh), urlStr, listener);
                    } else {
                        failTask(new Exception("editor is null"), urlStr, listener);
                    }
                } catch (IOException e) {
                    failTask(e, urlStr, listener);
                } catch (IllegalStateException e) {
                    failTask(e, urlStr, listener);
                }
            }
        };
    }

    private boolean downloadUrlToStream(String urlStr, DiskLruCache.Editor editor) {
        WritableByteChannel wbc = null;
        ReadableByteChannel rbc = null;
        try {
            wbc = Channels.newChannel(editor.newOutputStream(0));
            HttpURLConnection conn = (HttpURLConnection) new URL(urlStr).openConnection();
            rbc = Channels.newChannel(conn.getInputStream());
            ByteBuffer bb = ByteBuffer.allocate(1024);
            while (rbc.read(bb) != -1) {
                bb.flip();
                wbc.write(bb);
                bb.clear();
            }
            return true;
        } catch (final IOException e) {
            LogUtil.w("downloadUrlToStream", e);
        } finally {
            IOUtil.closeIO(wbc);
            IOUtil.closeIO(rbc);
        }
        return false;
    }

    @TargetApi(Build.VERSION_CODES.HONEYCOMB_MR1)
    private ImageData generateImageData(Context context, String urlStr, int[] wh) {
        String key = Md5Util.hashKeyForDisk(urlStr);
        DiskLruCache.Snapshot snapShot = null;
        try {
            snapShot = diskCache.get(key);
        } catch (IOException e) {
            LogUtil.w("generateImageData", e);
            return null;
        }
        BufferedInputStream bis = null;
        try {
            if (snapShot != null) {
                Drawable drawable = null;
                long length = 0;
                bis = new BufferedInputStream(snapShot.getInputStream(0));
                if (urlStr.toLowerCase().endsWith(".gif")) {
                    byte[] buf = new byte[bis.available() + 1];
                    bis.read(buf);
                    drawable = new GifDrawable(buf);
                    length = ((GifDrawable) drawable).getFrameByteCount();
                } else {
                    Bitmap bitmap = BitmapUtil.decodeSampledBitmapFromInputStream(bis, wh[0], wh[1]);
                    if (bitmap == null || context == null)
                        return null;
                    length = bitmap.getByteCount();
                    drawable = new BitmapDrawable(context.getResources(), bitmap);
                }
                drawable.setBounds(0, 0, wh[0], wh[1]);
                return new ImageData(drawable, length);
            }
        } catch (IOException e) {
            LogUtil.w("generateImageData", e);
        } finally {
            if (bis != null)
                try {
                    bis.close();
                } catch (IOException e) {
                }
        }
        return null;
    }

    public static ImageSpan toImageSpan(Drawable drawable, String urlStr, int verticalAlignment) {
        return urlStr.toLowerCase().endsWith(".gif") ? new GifImageSpan(drawable, verticalAlignment) : new RelativeImageSpan(drawable, verticalAlignment);
    }
}
