package com.easefun.polyvsdk;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import android.content.Context;
import android.os.Environment;
import android.os.StatFs;
import android.os.storage.StorageManager;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;

/**
 * 用于适配不同型号手机，反射获取SD卡路径和状态
 * @author TanQu 2016-5-11
 */
@SuppressWarnings("unused")
public class PolyvDevMountInfo {

	private static final String TAG = PolyvDevMountInfo.class.getSimpleName();

	private static final int ERROR = -1;

	// class name
	private final static String CLASS_NAME = "android.os.storage.StorageVolume";

	// method name
	private final static String METHOD_GET_VOLUME_LIST = "getVolumeList";
	private final static String METHOD_GET_VOLUME_STATE = "getVolumeState";
	private final static String METHOD_IS_REMOVABLE = "isRemovable";
	private final static String METHOD_GET_PATH = "getPath";

	private final static String MOUNTED = "mounted";

	private static PolyvDevMountInfo INSTANCE;

	private String mSDCardPath = null;

	// internal file path
	private ConcurrentLinkedQueue<String> mInternalPathList = new ConcurrentLinkedQueue<String>();
	// external file path
	private ConcurrentLinkedQueue<String> mExternalPathList = new ConcurrentLinkedQueue<String>();

	private ExecutorService mExecutor = null;
	
	private Context mContext = null;

	private PolyvDevMountInfo() {
		mExecutor = Executors.newSingleThreadExecutor();
	}

	public static PolyvDevMountInfo getInstance() {
		synchronized (PolyvDevMountInfo.class) {
			if (null == INSTANCE) {
				INSTANCE = new PolyvDevMountInfo();
			}
			return INSTANCE;
		}
	}

	@Override
	protected void finalize() throws Throwable {
		super.finalize();
		synchronized (PolyvDevMountInfo.class) {
			mInternalPathList.clear();
			mExternalPathList.clear();
			mExecutor.shutdown();
			INSTANCE = null;
		}
	}

	/**
	 * 初始化，初始化完成调用回调方法，才能调用其他方法
	 * @param context
	 * @param callback
	 */
	public void init(Context context, final OnLoadCallback callback) {
		mContext = context.getApplicationContext();
		mExecutor.execute(new Runnable() {
			@Override
			public void run() {
				executeInit(mContext, callback);
			}
		});
	}

	/**
	 * 是否有可移除的存储介质（例如 SD 卡）或内部（不可移除）存储可供使用
	 * @return 有返回true，无返回false
	 */
	public boolean isSDCardAvaiable() {
		return !mExternalPathList.isEmpty() || !mInternalPathList.isEmpty();
	}

	/**
	 * 获取SD卡路径，优先外部SD卡路径，然后内存SD卡路径，然后内部存储
	 * @return
	 */
	@Nullable
	public String getSDCardPath() {
		return mSDCardPath;
	}

	@SuppressWarnings("deprecation")
	public long getSDCardTotalSpace() {
		long totalSpace = 0;
		if (!TextUtils.isEmpty(mSDCardPath)) {
			StatFs sf = null;
			try {
				sf = new StatFs(mSDCardPath);
			} catch (Exception e) {
				Log.e(TAG, PolyvSDKUtil.getExceptionFullMessage(e, -1));
				return ERROR;
			}
			
			long blockSize = sf.getBlockSize();
			long total = sf.getBlockCount();
			totalSpace = total * blockSize / 1024;
		}
		
		return totalSpace;
	}

	@SuppressWarnings("deprecation")
	public long getSDCardAvailSpace() {
		long availSpace = 0;
		if (!TextUtils.isEmpty(mSDCardPath)) {
			StatFs sf = null;
			try {
				sf = new StatFs(mSDCardPath);
			} catch (Exception e) {
				Log.e(TAG, PolyvSDKUtil.getExceptionFullMessage(e, -1));
				return ERROR;
			}
			
			long blockSize = sf.getBlockSize();
			long availCount = sf.getAvailableBlocks();
			availSpace = availCount * blockSize / 1024;
		}
		
		return availSpace;
	}

	/**
	 * 取得内部（不可移除）存储介质路径
	 * @return 有则返回字符串，无则返回null
	 */
	@Nullable
	public String getInternalSDCardPath() {
		return mInternalPathList.peek();
	}

	/**
	 * 获取可移除的存储介质（例如 SD 卡）路径
	 * @return 有则返回字符串，无则返回null
	 */
	@Nullable
	public String getExternalSDCardPath() {
		return mExternalPathList.peek();
	}

	private void executeInit(Context context, OnLoadCallback callback) {
		StorageManager mStorageManager = (StorageManager) context.getSystemService(Context.STORAGE_SERVICE);
		if (mStorageManager != null) {
			Class<?> mStorageVolume;
			Method mGetVolumeListMethod;
			Method mGetVolumeStateMethod;
			Method mGetPathMethod;
			Method mIsRemovableMethod;
			Object[] mStorageVolumeList;
			try {
				mStorageVolume = Class.forName(CLASS_NAME);
				mGetVolumeListMethod = mStorageManager.getClass().getMethod(METHOD_GET_VOLUME_LIST, new Class[0]);
				mGetVolumeStateMethod = mStorageManager.getClass().getMethod(METHOD_GET_VOLUME_STATE, new Class[] { String.class });
				mIsRemovableMethod = mStorageVolume.getMethod(METHOD_IS_REMOVABLE, new Class[0]);
				mGetPathMethod = mStorageVolume.getMethod(METHOD_GET_PATH, new Class[0]);
				mStorageVolumeList = (Object[]) mGetVolumeListMethod.invoke(mStorageManager, new Object[0]);
				boolean mIsRemovable = false;
				if (mStorageVolumeList != null && mStorageVolumeList.length > 0) {
					int mStorageVolumeCount = mStorageVolumeList.length;
					Log.i(TAG, "init() === > StorageVolume Count = " + mStorageVolumeCount);

					mInternalPathList.clear();
					mExternalPathList.clear();

					for (Object aMStorageVolumeList : mStorageVolumeList) {
						String mStoragePath = (String) mGetPathMethod.invoke(aMStorageVolumeList, new Object[0]);
						mIsRemovable = ((Boolean) mIsRemovableMethod.invoke(aMStorageVolumeList, new Object[0])).booleanValue();
						if (!TextUtils.isEmpty(mStoragePath)) {
							String state = (String) mGetVolumeStateMethod.invoke(mStorageManager, new Object[]{mStoragePath});
							if ((state != null) && (state.equals(MOUNTED))) {
								if (mIsRemovable) {
									Log.i(TAG, "init() === > external storage path = (" + mStoragePath + ")");
									mExternalPathList.add(mStoragePath);
								} else {
									Log.i(TAG, "init() === > internal storage path = (" + mStoragePath + ")");
									mInternalPathList.add(mStoragePath);
								}
							}
						}
					}
				}
			} catch (ClassNotFoundException e) {
				handleInvalid();
				Log.e(TAG, "init() === > Exception:ClassNotFoundException");
			} catch (NoSuchMethodException e) {
				handleInvalid();
				Log.e(TAG, "init() === > Exception:NoSuchMethodException");
			} catch (IllegalArgumentException e) {
				handleInvalid();
				Log.e(TAG, "init() === > Exception:IllegalArgumentException");
			} catch (IllegalAccessException e) {
				handleInvalid();
				Log.e(TAG, "init() === > Exception:IllegalAccessException");
			} catch (InvocationTargetException e) {
				handleInvalid();
				Log.e(TAG, "init() === > Exception:InvocationTargetException");
			}
		} else {
			handleInvalid();
			Log.e(TAG, "init() === > can't get storage manager");
		}
		initSDCardPath();
		if (callback != null) {
			callback.callback();
		}
	}

	private void handleInvalid() {
		mInternalPathList.add(Environment.getExternalStorageDirectory().getPath());
	}

	private void initSDCardPath() {
		if (!mExternalPathList.isEmpty()) {
			mSDCardPath = mExternalPathList.peek();
		} else if (!mInternalPathList.isEmpty()) {
			mSDCardPath = mInternalPathList.peek();
		} else {
			mSDCardPath = Environment.getExternalStorageDirectory().getPath();
		}
		Log.i(TAG, "initSDCardPath() === > SDCARD PATH = (" + mSDCardPath + ")");
	}

	public static boolean externalMemoryAvailable() {
		return Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED);
	}

	public int getExternalPathListSize() {
		return mExternalPathList.size();
	}

	@SuppressWarnings("deprecation")
	public static long getAvailableInternalMemorySize() {
		File path = Environment.getDataDirectory();
		StatFs stat = null;
		try {
			stat = new StatFs(path.getPath());
		} catch (Exception e) {
			Log.e(TAG, PolyvSDKUtil.getExceptionFullMessage(e, -1));
			return ERROR;
		}
		
		long blockSize = stat.getBlockSize();
		long availableBlocks = stat.getAvailableBlocks();
		return availableBlocks * blockSize;
	}

	@SuppressWarnings("deprecation")
	public static long getTotalInternalMemorySize() {
		File path = Environment.getDataDirectory();
		StatFs stat = null;
		try {
			stat = new StatFs(path.getPath());
		} catch (Exception e) {
			Log.e(TAG, PolyvSDKUtil.getExceptionFullMessage(e, -1));
			return ERROR;
		}
		
		long blockSize = stat.getBlockSize();
		long totalBlocks = stat.getBlockCount();
		return totalBlocks * blockSize;
	}

	@SuppressWarnings("deprecation")
	public static long getAvailableInternalSystemMemorySize() {
		File path = Environment.getRootDirectory();
		StatFs stat = null;
		try {
			stat = new StatFs(path.getPath());
		} catch (Exception e) {
			Log.e(TAG, PolyvSDKUtil.getExceptionFullMessage(e, -1));
			return ERROR;
		}
		
		long blockSize = stat.getBlockSize();
		long availableBlocks = stat.getAvailableBlocks();
		return availableBlocks * blockSize;
	}

	@SuppressWarnings("deprecation")
	public static long getTotalInternalSystemMemorySize() {
		File path = Environment.getRootDirectory();
		StatFs stat = null;
		try {
			stat = new StatFs(path.getPath());
		} catch (Exception e) {
			Log.e(TAG, PolyvSDKUtil.getExceptionFullMessage(e, -1));
			return ERROR;
		}
		
		long blockSize = stat.getBlockSize();
		long totalBlocks = stat.getBlockCount();
		return totalBlocks * blockSize;
	}

	@SuppressWarnings("deprecation")
	public static long getAvailableExternalMemorySize() {
		if (externalMemoryAvailable()) {
			File path = Environment.getExternalStorageDirectory();
			StatFs stat = null;
			try {
				stat = new StatFs(path.getPath());
			} catch (Exception e) {
				Log.e(TAG, PolyvSDKUtil.getExceptionFullMessage(e, -1));
				return ERROR;
			}
			
			long blockSize = stat.getBlockSize();
			long availableBlocks = stat.getAvailableBlocks();
			return availableBlocks * blockSize;
		} else {
			return ERROR;
		}
	}

	@SuppressWarnings("deprecation")
	public static long getTotalExternalMemorySize() {
		if (externalMemoryAvailable()) {
			File path = Environment.getExternalStorageDirectory();
			StatFs stat = null;
			try {
				stat = new StatFs(path.getPath());
			} catch (Exception e) {
				Log.e(TAG, PolyvSDKUtil.getExceptionFullMessage(e, -1));
				return ERROR;
			}
			
			long blockSize = stat.getBlockSize();
			long totalBlocks = stat.getBlockCount();
			return totalBlocks * blockSize;
		} else {
			return ERROR;
		}
	}

	@SuppressWarnings("deprecation")
	public static long getAvailableMemorySize(String path) {
		if (null == path)
			return 0;
		StatFs stat = null;
		try {
			stat = new StatFs(path);
		} catch (Exception e) {
			Log.e(TAG, PolyvSDKUtil.getExceptionFullMessage(e, -1));
			return ERROR;
		}
		 
		long blockSize = stat.getBlockSize();
		long availableBlocks = stat.getAvailableBlocks();
		return availableBlocks * blockSize;
	}

	/**
	 * 创建目录
	 * @param dir 目录
	 * @return {@code true}:成功<br/>{@code false}:失败
	 */
	public boolean mkdirs(@NonNull File dir) throws IllegalArgumentException {
		 if (!dir.exists()) {
			 for (int i = 3 ; i > 0 ; i--) {
				 boolean value = executeMkdirs(dir);
				 if (value) {
					 return true;
				 }
				 
				 try {
					TimeUnit.MILLISECONDS.sleep(300);
				} catch (InterruptedException e) {
					Log.e(TAG, PolyvSDKUtil.getExceptionFullMessage(e, -1));
				}
			 }
		 } else {
			 return true;
		 }
		 
		 return false;
	}
	
	private boolean executeMkdirs(File dir) {
		if (!dir.exists()) {
			dir.mkdirs();
		}

		if (!dir.exists() && mContext != null) {
			mContext.getExternalFilesDir(null);
			dir.mkdirs();
		}
		
		return dir.exists();
	}
	
	public interface OnLoadCallback {
		void callback();
	}
}