package com.easefun.polyvsdk;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.SparseArray;

/**
 * 码率，本类静态方法封装了一些码率相关的方法，使用本方法统一取得码率相关信息
 * @version 2.0
 * @author TanQu 2015-9-29
 */
public enum PolyvBitRate {
	/** 自动 */
	ziDong(0) {
		@Override
		public String getName() {
			return ziDongName;
		}

		@Override
		public int getIndexByVideoUrlList() {
			return 0;
		}
	},
	/** 流畅 */
	liuChang(1) {
		@Override
		public String getName() {
			return liuChangName;
		}

		@Override
		public int getIndexByVideoUrlList() {
			return 0;
		}
	},
	/** 高清 */
	gaoQing(2) {
		@Override
		public String getName() {
			return gaoQingName;
		}

		@Override
		public int getIndexByVideoUrlList() {
			return 1;
		}
	},
	/** 超清 */
	chaoQing(3) {
		@Override
		public String getName() {
			return chaoQingName;
		}

		@Override
		public int getIndexByVideoUrlList() {
			return 2;
		}
	};
	
	private final int num;
	PolyvBitRate(int num) {
		this.num = num;
	}
	
	/**
	 * 取得码率对应的数字
	 * @return
	 */
	public int getNum() {
		return num;
	}
	
	/**
	 * 取得码率对应的名称
	 * @return
	 */
	public abstract String getName();
	public abstract int getIndexByVideoUrlList();
	
	/** 自动的名称，当没有选择码率播放时，播放器自动选择一个码率进行播放 */
	public static String ziDongName = "自动";
	/** 流程的名称 */
	public static String liuChangName = "流畅";
	/** 高清的名称 */
	public static String gaoQingName = "高清";
	/** 超清的名称 */
	public static String chaoQingName = "超清";
	
	/** dfNum码率个数对应的码率名称数组 */
	private static final SparseArray<String[]> nameArray;
	/** dfNum码率个数对应的码率列表 */
	private static final SparseArray<List<PolyvBitRate>> bitRateArray;
	/** 最低码率 */
	private static final PolyvBitRate minBitRate;
	/** 最高码率 */
	private static final PolyvBitRate maxBitRate;
	/** 倒序码率列表 */
	private static final List<PolyvBitRate> descBitRateList;
	
	static {
		//使用动态整理视频码率个数，以后维护只需要在上面增加常量就可以了，码率统一维护并且降低维护成本
		nameArray = new SparseArray<>();
		bitRateArray = new SparseArray<>();
		
		PolyvBitRate[] bitRateEnums = PolyvBitRate.values();
		bitRateEnums = Arrays.copyOfRange(bitRateEnums, 1, bitRateEnums.length);
		List<PolyvBitRate> bitRateEnumList = Arrays.asList(bitRateEnums);
		
		String[] names = null;
		List<PolyvBitRate> bitRateList= null;
		int length = bitRateEnumList.size();
		PolyvBitRate bitRateEnum = null;
		for (int i = 1 ; i <= length ; i++) {
			names = new String[i];
			bitRateList = new ArrayList<>(i);
			for (int j = 0 ; j < i ; j++) {
				bitRateEnum = bitRateEnumList.get(j);
				names[j] = bitRateEnum.getName();
				bitRateList.add(bitRateEnum);
			}
			
			nameArray.append(i, names);
			bitRateArray.append(i, bitRateList);
		}
		
		minBitRate = bitRateEnumList.get(0);
		maxBitRate = bitRateEnumList.get(length - 1);
		
		//反转码率列表，倒序排序
		Collections.reverse(bitRateEnumList);
		descBitRateList = bitRateEnumList;
	}
	
	/**
	 * 取得最低使用的码率(不包括{@link #ziDong})
	 * @return
	 */
	public static PolyvBitRate getMinBitRate() {
		return minBitRate;
	}
	
	/**
	 * 取得最低使用的码率(包括{@link #ziDong})
	 * @return
	 */
	public static PolyvBitRate getMinBitRateFromAll() {
		return ziDong;
	}
	
	/**
	 * 取得最高使用的码率
	 * @return
	 */
	public static PolyvBitRate getMaxBitRate() {
		return maxBitRate;
	}
	
	/**
	 * 根据码率个数取得当中最高可以使用的码率
	 * @param dfNum 码率个数
	 * @return
	 */
	public static PolyvBitRate getMaxBitRate(int dfNum) {
		List<PolyvBitRate> list = bitRateArray.get(dfNum);
		return list.get(list.size() - 1);
	}
	
	/**
	 * 根据码率个数取得对应的码率名称数组
	 * @param dfNum 码率个数
	 * @return
	 */
	public static String[] getBitRateNameArray(int dfNum) {
		return nameArray.get(dfNum);
	}
	
	/**
	 * 根据码率个数取得对应的码率列表
	 * @param dfNum 码率个数
	 * @return
	 */
	public static List<PolyvBitRate> getBitRateList(int dfNum) {
		return bitRateArray.get(dfNum);
	}
	
	/**
	 * 根据码率取得BitRateEnum 成员
	 * @param bitRate 码率
	 * @return
	 */
	@Nullable
	public static PolyvBitRate getBitRate(int bitRate) {
		return getBitRate(bitRate, null);
	}

	/**
	 * 根据码率取得BitRateEnum 成员
	 * @param bitRate 码率
     * @param defaultBitRate 默认返回
	 * @return
	 */
	public static PolyvBitRate getBitRate(int bitRate, PolyvBitRate defaultBitRate) {
		PolyvBitRate[] array = PolyvBitRate.values();
		for (PolyvBitRate bitRateEnum : array) {
			if (bitRateEnum.getNum() == bitRate) return bitRateEnum;
		}

		return defaultBitRate;
	}
	
	/**
	 * 取得码率名称
	 * @param bitRate 码率
	 * @return 码率名称
	 */
	@Nullable
	public static String getBitRateName(int bitRate) {
		PolyvBitRate bitRateEnum = getBitRate(bitRate);
		if (bitRateEnum == null) return "";
		return bitRateEnum.getName();
	}
	
	/**
	 * 取得倒序码率列表
	 * @return 码率列表
	 */
	@NonNull
	public static List<PolyvBitRate> getDescBitRateList() {
		return getDescBitRateList(true);
	}

    /**
     * 取得倒序码率列表
     * @param isIncludeZiDong 是否包含{@link #ziDong}
     * @return 码率列表
     */
    @NonNull
    public static List<PolyvBitRate> getDescBitRateList(boolean isIncludeZiDong) {
        if (isIncludeZiDong) {
            return descBitRateList;
        }

        List<PolyvBitRate> list = new ArrayList<>();
        for (PolyvBitRate bitRate : descBitRateList) {
            if (bitRate == ziDong) continue;
            list.add(bitRate);
        }

        return list;
    }
}
