/*
 * Copyright 2015 Dynatrace
 *
 * Licensed under the Dynatrace SaaS terms of service (the "License");
 * You may obtain a copy of the License at
 *
 *      https://ruxit.com/eula/saas/#terms-of-service
 */
package com.dynatrace.tools.android;

import static com.dynatrace.tools.android.AndroidPluginVersion.VERSION_1_5;
import static com.dynatrace.tools.android.AndroidPluginVersion.VERSION_2_2;
import static com.dynatrace.tools.android.AndroidPluginVersion.VERSION_2_3;
import static com.dynatrace.tools.android.AndroidPluginVersion.VERSION_3_0;
import static com.dynatrace.tools.android.AndroidPluginVersion.VERSION_3_1;
import static com.dynatrace.tools.android.AndroidPluginVersion.VERSION_3_3;
import static com.google.common.base.Preconditions.checkNotNull;

import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.function.BooleanSupplier;
import java.util.function.Predicate;

import org.gradle.api.GradleException;
import org.gradle.api.Project;
import org.gradle.api.file.FileTree;
import org.gradle.api.internal.ConventionTask;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.InputFile;
import org.gradle.api.tasks.InputFiles;
import org.gradle.api.tasks.Internal;
import org.gradle.api.tasks.Optional;
import org.gradle.api.tasks.OutputFile;
import org.gradle.api.tasks.TaskAction;

import com.android.build.gradle.tasks.PackageApplication;
import com.android.builder.core.AndroidBuilder;
import com.android.builder.model.SigningConfig;
import com.android.ide.common.signing.CertificateInfo;
import com.android.ide.common.signing.KeystoreHelper;

/**
 * Task to modify the APK file.
 */
public class AutoInstrumentTask extends ConventionTask {
	/**
	 * APK file to modify
	 */
	private File apkFile;

	/**
	 * Application id in dynatrace
	 */
	private String applicationId;

	/**
	 * Environment id in dynatrace
	 */
	private String environmentId;

	/**
	 * Cluster to report data to
	 */
	private String cluster;

	/**
	 * Environment id in dynatrace
	 */
	private String startupPath;

	/**
	 * beaconURL id in dynatrace
	 */
	private String beaconURL;

	/**
	 * Additional agent properties
	 */
	private Map<String, String> agentProperties;

	private SigningConfig signingConfig;

	private File apkitDir;

	/**
	 * The instrumented APK
	 */
	private File outputFile;

	private AndroidPluginVersion androidPluginVersion = VERSION_1_5;
	private PackageApplication packageTask;

	@TaskAction
	public void instrument() {


		Properties tmpProperties = new Properties();
		tmpProperties.putAll(getAgentProperties());

		if (getApplicationId() != null) {
			tmpProperties.put("DTXApplicationID", getApplicationId());
		}

		boolean useBeaconURL = getBeaconURL() != null;

		if (useBeaconURL) {
			tmpProperties.put("DTXBeaconURL", getBeaconURL());
		} else {
			boolean isAppMon = getEnvironmentId() == null;
			if (!isAppMon) {
				if (getEnvironmentId() != null) {
					tmpProperties.put("DTXAgentEnvironment", getEnvironmentId());
				}
				if (getCluster() != null) {
					tmpProperties.put("DTXClusterURL", getCluster());
				}
			} else {
				if (getStartupPath() != null) {
					tmpProperties.put("DTXAgentStartupPath", getStartupPath());
				}
			}
		}

		final File propertyFile = new File(getTemporaryDir(), "instrument.properties");
		try (BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(propertyFile))) {
			tmpProperties.store(out, "Autogenerated properties");
		} catch (IOException ex) {
			throw new GradleException("Exception writing instrument.properties", ex);
		}

		final String executableName = System.getProperty("os.name").toLowerCase().contains("windows") ?
				"instrument.cmd" :
				"instrument.sh";

		final File instrumentExecutable = getProject().fileTree(getApkitDir())
				.filter(file -> executableName.equalsIgnoreCase(file.getName())).getSingleFile();

		instrumentExecutable.setExecutable(true);

		if (getApkFile() == null || !getApkFile().getName().endsWith(".apk")) {
			throw new GradleException("Task input contains an invalid apk file: " + getApkFile());
		}

		String dirName = getApkFile().getName().substring(0, getApkFile().getName().lastIndexOf(".apk"));
		File instrumentDist = new File(getApkFile().getParentFile(), dirName + "/dist");
		File instrumentDirectory = new File(getApkFile().getParentFile(), dirName);
		File instrumentedAPK = new File(instrumentDist, getApkFile().getName());

		// clean up instrumentation files
		if (instrumentDirectory.exists()) {
			getProject().delete(instrumentDirectory);
		}

		getProject().exec(execSpec -> {
			execSpec.setExecutable(instrumentExecutable);
			execSpec.setWorkingDir((Object) getTemporaryDir());
			execSpec.args("apk=" + getApkFile().getAbsolutePath(), "prop=" + propertyFile.getAbsolutePath());
		});

		if (!instrumentedAPK.exists()) {
			throw new IllegalStateException("auto-instrumentation failed");
		}

		try {

			AndroidBuilder androidBuilder;
			String methodBuilderClass = androidPluginVersion.isOlderThan(VERSION_3_1) ?
					"com.android.build.gradle.internal.tasks.BaseTask" :
					"com.android.build.gradle.internal.tasks.AndroidBuilderTask";

			Method methodBuilder = Class.forName(methodBuilderClass).getDeclaredMethod("getBuilder");
			methodBuilder.setAccessible(true);
			androidBuilder = (AndroidBuilder) methodBuilder.invoke(packageTask);

			if (androidPluginVersion == VERSION_1_5) {
				Method methodSignApk = AndroidBuilder.class
						.getDeclaredMethod("signApk", File.class, SigningConfig.class, File.class);
				methodSignApk.setAccessible(true);
				methodSignApk.invoke(androidBuilder, instrumentedAPK, getSigningConfig(), getOutputFile());
			} else {
				String mainFileName = getApkFile().getAbsolutePath();
				String origFileName = mainFileName.substring(0, mainFileName.length() - 4) + "_uninstrumented.apk";
				getApkFile().renameTo(new File(origFileName));

				Class classPackageAndroidArtifact = Class.forName("com.android.build.gradle.tasks.PackageAndroidArtifact");

				Method methodMinSdk = classPackageAndroidArtifact.getDeclaredMethod("getMinSdkVersion");
				methodMinSdk.setAccessible(true);
				int minSdkVersion = (int) methodMinSdk.invoke(packageTask);

				Method methodDebugBuild = classPackageAndroidArtifact.getDeclaredMethod("getDebugBuild");
				methodDebugBuild.setAccessible(true);
				boolean debugBuild = (boolean) methodDebugBuild.invoke(packageTask);

				Method methodCreatedBy = AndroidBuilder.class.getDeclaredMethod("getCreatedBy");
				methodCreatedBy.setAccessible(true);
				String createdBy = (String) methodCreatedBy.invoke(androidBuilder);

				PrivateKey key;
				X509Certificate certificate;
				boolean v1SigningEnabled;
				boolean v2SigningEnabled;

				if (signingConfig != null && signingConfig.isSigningReady()) {
					CertificateInfo certificateInfo =
							KeystoreHelper.getCertificateInfo(
									signingConfig.getStoreType(),
									checkNotNull(signingConfig.getStoreFile()),
									checkNotNull(signingConfig.getStorePassword()),
									checkNotNull(signingConfig.getKeyPassword()),
									checkNotNull(signingConfig.getKeyAlias()));
					key = certificateInfo.getKey();
					certificate = certificateInfo.getCertificate();

					Method methodIsV1 = SigningConfig.class.getMethod("isV1SigningEnabled");
					methodIsV1.setAccessible(true);
					Method methodIsV2 = SigningConfig.class.getMethod("isV2SigningEnabled");
					methodIsV2.setAccessible(true);

					v1SigningEnabled = (boolean) methodIsV1.invoke(signingConfig);
					v2SigningEnabled = (boolean) methodIsV2.invoke(signingConfig);
				} else {
					key = null;
					certificate = null;
					v1SigningEnabled = false;
					v2SigningEnabled = false;
				}

				String packageName;
				if (androidPluginVersion.isOlderOrEqualTo(VERSION_2_2)) {
					packageName = "com.android.builder.packaging.";
				} else if (androidPluginVersion.isOlderOrEqualTo(VERSION_3_1)) {
					packageName = "com.android.apkzlib.zfile.";
				} else { // from Version_3_2 onwards
					packageName = "com.android.tools.build.apkzlib.zfile.";
				}

				Class classNativeLibrariesPackagingMode = Class.forName(packageName + "NativeLibrariesPackagingMode");
				Class classCreationData = Class.forName(packageName + "ApkCreatorFactory$CreationData");
				Constructor constructor = getCreationDataConstructor(classNativeLibrariesPackagingMode,
						classCreationData);

				Class classPackagingUtils = Class.forName("com.android.builder.packaging.PackagingUtils");
				Method methodNativeLibraries = null;
				if (androidPluginVersion.isOlderOrEqualTo(VERSION_3_1)) {
					methodNativeLibraries = classPackagingUtils
							.getDeclaredMethod("getNativeLibrariesLibrariesPackagingMode", File.class);
				} else { // from Version_3_2 onwards
					Class evalIssueReporter = Class.forName("com.android.builder.errors.EvalIssueReporter");
					methodNativeLibraries = classPackagingUtils
							.getDeclaredMethod("getNativeLibrariesLibrariesPackagingMode", File.class, BooleanSupplier.class,
									evalIssueReporter);
				}

				methodNativeLibraries.setAccessible(true);

				Object nativeLibrariesPackagingMode = getNativeLibrariesPackagingMode(dirName, classPackageAndroidArtifact,
						methodNativeLibraries);
				Object noCompressPredicate = getCompressionPredicate(classPackageAndroidArtifact, classNativeLibrariesPackagingMode,
						classPackagingUtils,
						nativeLibrariesPackagingMode);

				Object creationData = constructor.newInstance(
						getOutputFile(),
						key,
						certificate,
						v1SigningEnabled,
						v2SigningEnabled,
						null, // BuiltBy
						createdBy,
						minSdkVersion,
						nativeLibrariesPackagingMode,
						noCompressPredicate);

				Class classApkCreatorFactories = Class.forName("com.android.build.gradle.internal.packaging.ApkCreatorFactories");
				Method methodFromProjectProperties = classApkCreatorFactories
						.getDeclaredMethod("fromProjectProperties", Project.class, boolean.class);
				methodFromProjectProperties.setAccessible(true);
				Object factory = methodFromProjectProperties.invoke(null, getProject(), debugBuild);

				Class classApkCreatorFactory = Class.forName(packageName + "ApkCreatorFactory");
				Method methodMake = classApkCreatorFactory.getDeclaredMethod("make", classCreationData);
				methodMake.setAccessible(true);

				try (Closeable creator = (Closeable) methodMake.invoke(factory, creationData)) {
					Method methodWriteZip = getWriteZipMethod(creator.getClass());
					methodWriteZip.setAccessible(true);
					methodWriteZip.invoke(creator, instrumentedAPK, null, null);
				}
			}
		} catch (Exception ex) {
			throw new GradleException("Failed to sign apk", ex);
		}
	}

	private Object getNativeLibrariesPackagingMode(String dirName, Class classPackageAndroidArtifact, Method methodNativeLibraries)
			throws IllegalAccessException, InvocationTargetException, NoSuchFieldException, NoSuchMethodException {

		Object nativeLibrariesPackagingMode;

		if (androidPluginVersion.isNewerOrEqualTo(VERSION_3_0)) {
			//TODO split feature not supported: see ONE-9418
			File origManifestFile = new File(getApkFile().getParentFile(), dirName + "/AndroidManifest-orig.xml");

			if (androidPluginVersion.isOlderOrEqualTo(VERSION_3_1)) {
				nativeLibrariesPackagingMode = methodNativeLibraries.invoke(null, origManifestFile);
			} else { // from Version_3_2 onwards
				// InExecutionPhase is always true:
				BooleanSupplier execPhase = () -> true;
				// last parameter (EvalIssueReporter) can be null since we need no reports:
				nativeLibrariesPackagingMode = methodNativeLibraries.invoke(null, origManifestFile, execPhase, null);
			}

		} else {
			Field fieldManifest = classPackageAndroidArtifact.getDeclaredField("manifest");
			fieldManifest.setAccessible(true);
			File manifest = (File) fieldManifest.get(packageTask);

			nativeLibrariesPackagingMode = methodNativeLibraries.invoke(null, manifest);

			Method methodGetNoCompressPredicate = classPackageAndroidArtifact.getDeclaredMethod("getNoCompressPredicate");
			methodGetNoCompressPredicate.setAccessible(true);
		}
		return nativeLibrariesPackagingMode;
	}

	private Object getCompressionPredicate(Class classPackageAndroidArtifact, Class classNativeLibrariesPackagingMode,
			Class classPackagingUtils, Object nativeLibrariesPackagingMode)
			throws NoSuchMethodException, IllegalAccessException, InvocationTargetException {
		Object noCompressPredicate;
		if (androidPluginVersion.isNewerOrEqualTo(VERSION_3_0)) {
			// getting a list of extensions defined via aaptOptions{ noCompress ...} that should not be compressed
			Method methodGetNoCompressExtensions = classPackageAndroidArtifact.getDeclaredMethod("getNoCompressExtensions");
			methodGetNoCompressExtensions.setAccessible(true);
			Collection<String> noCompEx = (Collection<String>) methodGetNoCompressExtensions.invoke(packageTask);

			// also take the packaging mode and the default compression exceptions into account
			Method methodGetAllNoCompressExtensions = classPackagingUtils
					.getDeclaredMethod("getAllNoCompressExtensions", Collection.class, classNativeLibrariesPackagingMode);
			methodGetAllNoCompressExtensions.setAccessible(true);
			List<String> noCompressExtensions = (List<String>) methodGetAllNoCompressExtensions
					.invoke(null, noCompEx, nativeLibrariesPackagingMode);
			// this list now contains all default extensions that should not be compressed + the ones from the noCompEx list and *.so depending whether the NativeLibrariesPackagingMode says so or not

			// getting the predicate
			Method methodGetNoCompressPredicateForExtension = classPackagingUtils
					.getDeclaredMethod("getNoCompressPredicateForExtensions", Iterable.class);
			methodGetNoCompressPredicateForExtension.setAccessible(true);

			noCompressPredicate = methodGetNoCompressPredicateForExtension.invoke(null, noCompressExtensions);

			if (androidPluginVersion.isNewerOrEqualTo(VERSION_3_3)) {
				noCompressPredicate = (com.google.common.base.Predicate<String>) ((Predicate<String>)noCompressPredicate)::test;
			}
		} else { // plugin earlier than 3.0
			Method methodGetNoCompressPredicate = classPackageAndroidArtifact.getDeclaredMethod("getNoCompressPredicate");
			methodGetNoCompressPredicate.setAccessible(true);

			if (androidPluginVersion == VERSION_2_3) {
				noCompressPredicate = methodGetNoCompressPredicate.invoke(packageTask);
			} else {
				noCompressPredicate = ((Predicate<String>) ((com.google.common.base.Predicate<String>) methodGetNoCompressPredicate
						.invoke(packageTask))::apply);
			}
		}
		return noCompressPredicate;
	}

	private Constructor getCreationDataConstructor(Class classNativeLibrariesPackagingMode, Class classCreationData)
			throws NoSuchMethodException {
		Class predicateClass = java.util.function.Predicate.class;
		if (androidPluginVersion.isNewerOrEqualTo(VERSION_3_3)) {
			predicateClass = com.google.common.base.Predicate.class;
		}
		return classCreationData
				.getDeclaredConstructor(File.class, PrivateKey.class, X509Certificate.class, boolean.class, boolean.class,
						String.class, String.class, int.class, classNativeLibrariesPackagingMode,
						predicateClass);
	}

	private Method getWriteZipMethod(Class clazz) throws NoSuchMethodException {
		Class predicateClass = java.util.function.Predicate.class;
		Class functionClass = java.util.function.Function.class;

		if (androidPluginVersion.isNewerOrEqualTo(VERSION_3_3)) {
			functionClass = com.google.common.base.Function.class;
			predicateClass = com.google.common.base.Predicate.class;
		}
		return clazz.getDeclaredMethod("writeZip", File.class, functionClass,
				predicateClass);
	}

	@InputFile
	public File getApkFile() {
		return apkFile;
	}

	public void setApkFile(File apkFile) {
		this.apkFile = apkFile;
	}

	@Input
	@Optional
	public String getApplicationId() {
		return applicationId;
	}

	public void setApplicationId(String applicationId) {
		this.applicationId = applicationId;
	}

	@Input
	@Optional
	public String getEnvironmentId() {
		return environmentId;
	}

	public void setEnvironmentId(String environmentId) {
		this.environmentId = environmentId;
	}

	@Input
	@Optional
	public String getCluster() {
		return cluster;
	}

	public void setCluster(String cluster) {
		this.cluster = cluster;
	}

	@Input
	@Optional
	public String getBeaconURL() {
		return beaconURL;
	}

	public void setBeaconURL(String beaconURL) {
		this.beaconURL = beaconURL;
	}

	@Input
	@Optional
	public String getStartupPath() {
		return startupPath;
	}

	public void setStartupPath(String startupPath) {
		this.startupPath = startupPath;
	}

	@Input
	public Map<String, String> getAgentProperties() {
		return agentProperties;
	}

	public void setAgentProperties(Map<String, String> agentProperties) {
		this.agentProperties = agentProperties;
	}

	public SigningConfig getSigningConfig() {
		return signingConfig;
	}

	public void setSigningConfig(SigningConfig signingConfig) {
		this.signingConfig = signingConfig;
	}

	@Internal
	//@InputDirectory // after dropping gradle 2.x support
	public File getApkitDir() {
		return apkitDir;
	}

	public void setApkitDir(File apkitDir) {
		this.apkitDir = apkitDir;
	}

	// only used for gradle 2.x compatible up-to-date check of directory contents
	// method can be removed and replaced with @InputDirectory on the plain getter after dropping 2.x support
	@Optional
	@InputFiles
	FileTree getApkitFiles() {
		if (apkitDir != null) {
			return getProject().fileTree(apkitDir);
		}
		return null;
	}

	@OutputFile
	public File getOutputFile() {
		return outputFile;
	}

	public void setOutputFile(File outputFile) {
		this.outputFile = outputFile;
	}

	@Input
	public AndroidPluginVersion getAndroidPluginVersion() {
		return androidPluginVersion;
	}

	public void setAndroidPluginVersion(AndroidPluginVersion androidPluginVersion) {
		this.androidPluginVersion = androidPluginVersion;
	}

	public PackageApplication getPackageTask() {
		return packageTask;
	}

	public void setPackageTask(PackageApplication packageTask) {
		this.packageTask = packageTask;
	}
}
