package com.dada.smart.user;

import android.app.Activity;
import android.support.annotation.VisibleForTesting;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewParent;
import android.view.accessibility.AccessibilityEvent;

import com.dada.smart.common.ResourceIds;
import com.dada.smart.common.Utils;
import com.dada.smart.common.WindowMagician;
import com.dada.smart.user.config.ClickEvents;
import com.dada.smart.user.config.Event;
import com.orhanobut.logger.Logger;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * Created by tomkeyzhang on 21/5/18.
 */
public class ViewVisitor {

    private ResourceIds resourceIds;
    private OnEventListener listener;
    private Map<View, AccessibilityDelegate> delegateMap;
    private Executor executor = new ThreadPoolExecutor(1, 1, 0, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(64), new ThreadPoolExecutor.DiscardOldestPolicy());
    //本界面绑定的点击事件
    private ClickEvents clickEvents;
    private List<Event> events;
    private WeakReference<Activity> weakReference;
    private EventLogRepository eventLogRepository;
    private String activityName;


    public ViewVisitor(ResourceIds resourceIds, Activity activity, EventLogRepository eventLogRepository, OnEventListener listener) {
        this.resourceIds = resourceIds;
        weakReference = new WeakReference<>(activity);
        clickEvents = eventLogRepository.getEvents(Utils.canonicalName(activity));
        if (clickEvents != null) {
            events = clickEvents.getEvents();
        } else {
            this.eventLogRepository = eventLogRepository;
            activityName = Utils.canonicalName(activity);
        }
        this.listener = listener;
        this.delegateMap = new HashMap<>();
    }

    public interface OnEventListener {
        void onEvent(Event event);
    }

    public void bindEvents() {
        if ((events == null || events.isEmpty()) && eventLogRepository != null && !TextUtils.isEmpty(activityName)) {
            clickEvents = eventLogRepository.getEvents(Utils.canonicalName(activityName));
            if (clickEvents != null) {
                events = clickEvents.getEvents();
            }
        }
        if (events != null && !events.isEmpty()) {
            executor.execute(new Runnable() {
                @Override
                public void run() {
                    try {
                        Thread.sleep(200);
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                    Activity activity = weakReference.get();
                    if (activity != null) {
                        delegateMap.clear();
                        if (clickEvents.isInMainWindow()) {
                            bindEventToTargetView(activity.getWindow().getDecorView(), events, 0);
                        } else {
                            List<WindowMagician.ViewRootData> list = WindowMagician.getFocusWindowRootView(activity);
                            for (WindowMagician.ViewRootData viewRootData : list) {
                                bindEventToTargetView(viewRootData.getView(), events, 0);
                            }
                        }
                    }
                }
            });
        }
    }

    @VisibleForTesting
    public Map<View, AccessibilityDelegate> getDelegateMap() {
        return delegateMap;
    }

    /**
     * @param root
     * @param events
     * @param viewTreeLevel 视图层级中的第几层
     */
    private void bindEventToTargetView(View root, List<Event> events, int viewTreeLevel) {
        //        Logger.v("bindEventToTargetView:" + root);
        if (root == null)
            return;
        for (final Event event : events) {
            if (matches(root, event, viewTreeLevel)) {
                Logger.v("find view=" + root + " event" + event);
                addEventToAccessibilityDelegate(root, event);
            }
        }
        if (root instanceof ViewGroup) {
            ViewGroup parent = (ViewGroup) root;
            for (int i = 0; i < parent.getChildCount(); i++) {
                bindEventToTargetView(parent.getChildAt(i), events, viewTreeLevel + 1);
            }
        }
    }

    private void addEventToAccessibilityDelegate(View root, Event event) {
        AccessibilityDelegate delegate = delegateMap.get(root);
        if (delegate == null) {
            delegate = new AccessibilityDelegate();
            root.setAccessibilityDelegate(delegate);
            delegateMap.put(root, delegate);
        }
        delegate.addEvent(event);
    }

    /**
     * @param root
     * @param event
     * @param viewTreeLevel 视图层级中的第几层
     * @return
     */
    private boolean matches(View root, Event event, int viewTreeLevel) {
        List<Event.ViewTreeChild> list = event.getViewTreeChildren();
        if (list.isEmpty()) {
            String viewIdName = event.getViewIdName();
            String viewClassName = event.getViewClassName();
            if (viewClassName == null || viewIdName == null)
                return false;
            return viewIdName.equals(resourceIds.nameForId(root.getId())) && viewClassName.equals(Utils.canonicalName(root));
        } else {
            ViewParent parent = root.getParent();
            if (parent != null) {
                if (parent instanceof ViewGroup) {
                    if (viewTreeLevel < list.size()) {
                        Event.ViewTreeChild child = list.get(viewTreeLevel);
                        int childCount = ((ViewGroup) parent).getChildCount();
                        int index = ((ViewGroup) parent).indexOfChild(root);
                        if (TextUtils.equals(child.getClassName(), Utils.simpleName(root)) && childCount == child.getBrothersCount() && index == child.getIndex()) {
                            child.setPassVerify(true);
                        }
                    }
                } else {
                    //parent不是ViewGroup的情况只有ViewRootImpl一种，也就是DecorView的parent是ViewRootImpl
                    list.get(0).setPassVerify(true);
                }
            }
            return event.checkAllTreeViewVerified();
        }
    }

    private class AccessibilityDelegate extends View.AccessibilityDelegate {
        List<Event> events;

        public AccessibilityDelegate() {
            this.events = new CopyOnWriteArrayList<>();
        }

        public void addEvent(Event event) {
            events.add(event);
        }

        @Override
        public void sendAccessibilityEvent(View host, int eventType) {
            if (listener == null)
                return;
            //https://bugly.qq.com/v2/crash-reporting/crashes/9d9b1e7492/192272?pid=1
            //# main(1)
//            java.util.ConcurrentModificationException
//                    解析原始
//            1 java.util.ArrayList$Itr.next(ArrayList.java:831)
//            2 com.dada.smart.user.g$a.sendAccessibilityEvent(ViewVisitor.java:181)
            for (Event event : events) {
                if (matchEvent(event, eventType)) {
                    listener.onEvent(event);
                }
            }
        }

        boolean matchEvent(Event event, int eventType) {
            return Event.TYPE_CLICK == event.getTypeId() && eventType == AccessibilityEvent.TYPE_VIEW_CLICKED;
        }
    }

    @VisibleForTesting
    public void setExecutor(Executor executor) {
        this.executor = executor;
    }
}
