package com.dada.smart.user;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.support.annotation.VisibleForTesting;
import android.view.View;
import android.view.ViewTreeObserver;

import com.dada.smart.common.ActivityMonitor;
import com.dada.smart.common.SmartExpression;
import com.dada.smart.common.Utils;
import com.dada.smart.user.config.Event;
import com.dada.smart.user.http.CacheHttpClient;
import com.dada.smart.user.http.Client;
import com.dada.smart.user.log.LogDatabase;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

/**
 * Created by tomkeyzhang on 16/5/18.
 */

public class ViewMonitor extends ActivityMonitor {

    private ViewVisitor viewVisitor;
    private EventLogRepository eventLogRepository;
    private ViewVisitor.OnEventListener listener;
    private long appId;
    private SharedPreferences preferences;

    private ViewTreeObserver viewTreeObserver;


    public ViewMonitor(Context context, long appId, String server, Client client, EventLogRepository.RequestParam requestParam, boolean debug) {
        super(context, debug);
        this.appId = appId;
        this.preferences = context.getSharedPreferences("app_start.pref", Context.MODE_PRIVATE);
        this.eventLogRepository = new EventLogRepository.Builder(server).logDao(LogDatabase.getDatabase(context).logDao()).client(client).requestParam(requestParam).build();
        this.listener = new EventListener();
    }

    public ViewMonitor(Context context, long appId, String server, EventLogRepository.RequestParam requestParam, boolean debug) {
        this(context, appId, server, new CacheHttpClient(context.getSharedPreferences("manager.pref", Context.MODE_PRIVATE)), requestParam, debug);
    }


    public void update() {
        eventLogRepository.fetchConfig(appId);
        eventLogRepository.sendAppStartEventIfNeed(preferences, System.currentTimeMillis(), appId);
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
        super.onActivityCreated(activity, savedInstanceState);
        triggerPvEvent(Utils.canonicalName(activity), getLastPageClass());
        View view = activity.findViewById(android.R.id.content);
        viewVisitor = new ViewVisitor(getResourceIds(), activity, eventLogRepository, listener);
        viewTreeObserver = view.getViewTreeObserver();
        viewTreeObserver.addOnWindowFocusChangeListener(onWindowFocusChangeListener);
    }

    private ViewTreeObserver.OnWindowFocusChangeListener onWindowFocusChangeListener = new ViewTreeObserver.OnWindowFocusChangeListener() {
        @Override
        public void onWindowFocusChanged(boolean hasFocus) {
            if (viewVisitor != null) {
                viewVisitor.bindEvents();
            }
        }
    };

    @Override
    public void onActivityDestroyed(Activity activity) {
        super.onActivityDestroyed(activity);
        if (viewTreeObserver != null) {
            viewTreeObserver.removeOnWindowFocusChangeListener(onWindowFocusChangeListener);
        }
    }

    @Override
    public void onActivityStopped(Activity activity) {
        super.onActivityStopped(activity);
        //只有当前activity有配置过事件，在销毁的时候才进行一次日志发送
        if (eventLogRepository.hasEvent(Utils.canonicalName(activity)))
            sendEvents();
    }

    private void sendEvents() {
        eventLogRepository.sendEvents(appId);
    }

    private void triggerPvEvent(String activityName, String refPageName) {
        List<Event> events = eventLogRepository.getPvEvents(activityName);
        if (!Utils.isEmpty(events)) {
            for (Event event : events) {
                onEvent(event, refPageName);
            }
        }
    }

    private class EventListener implements ViewVisitor.OnEventListener {

        @Override
        public void onEvent(Event event) {
            ViewMonitor.this.onEvent(event, getLastPageClass());
//            eventLogRepository.onEvent(event.getId(), event.getTypeId(), getLastPageClass(), SmartExpression.fetchResult(uiThreadObject().get(), event.getExpressions()).toString());
        }
    }

    public void onEvent(Event event, String refPageName) {
        List<SmartExpression.Result> results = SmartExpression.fetchResult(uiThreadObject().get(), event.getExpressions());
        JSONObject object = new JSONObject();
        for (SmartExpression.Result result : results) {
            if (result.isTypeEqual()) {
                if (!result.isLeftEqualsValue())
                    return;//在相等的表达式类型下，只有当右侧表达式的值等于左侧才会触发事件
            } else if (SmartExpression.TYPE_ASSIGN.equals(result.getType())) {
                try {
                    object.put(result.getLeft(), result.getValue());
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
        }
        eventLogRepository.onEvent(event.getId(), event.getTypeId(), refPageName, object.toString());
    }

    @VisibleForTesting
    public void setListener(ViewVisitor.OnEventListener listener) {
        this.listener = listener;
    }

    @VisibleForTesting
    public void setPreferences(SharedPreferences preferences) {
        this.preferences = preferences;
    }


}
