package com.coder.mario.android.widget;

import android.content.Context;
import android.database.Observable;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.View;
import android.view.ViewGroup;

/**
 * Created by CoderMario on 2018-01-31.
 */
public class TagsLayout extends ViewGroup {

    /**
     * */
    private Adapter mAdapter;

    /**
     * */
    private final int SPACING_ORIENTATION_H = dp2value(6);

    /**
     * */
    private final int SPACING_ORIENTATION_V = dp2value(10);

    /**
     * */
    private int mSpacingOrientationV = SPACING_ORIENTATION_V;

    /**
     * */
    private int mSpacingOrientationH = SPACING_ORIENTATION_H;

    /**
     * */
    private final TagsLayoutDataObserver mObserver = new TagsLayoutDataObserver();

    public TagsLayout(Context context) {
        super(context);
        init();
    }

    public TagsLayout(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    public TagsLayout(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    private void init() {
        //
    }

    @Override
    protected LayoutParams generateLayoutParams(LayoutParams params) {
        return new MarginLayoutParams(params);
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
        int spacingOrientationV = getSpacingOrientationV();
        int spacingOrientationH = getSpacingOrientationH();

        int width = getMeasuredWidth();
        width -= getPaddingLeft() + getPaddingRight();

        int totalHeight = getPaddingTop(), lineWidth = 0, lineHeight = 0;
        for (int i = 0, size = getChildCount(); i < size; i ++) {
            View child = getChildAt(i);
            if (null == child || View.GONE == child.getVisibility()) {
                continue;
            }
            measureChild(child, widthMeasureSpec, heightMeasureSpec);

            int childHeight = child.getMeasuredHeight() ;
            int childWidth = child.getMeasuredWidth() ;

            if (lineWidth + childWidth <= width) {
                lineHeight = Math.max(lineHeight, childHeight);
                lineWidth += childWidth + spacingOrientationH;
            } else {
                totalHeight += lineHeight + (0 == i ? 0 : spacingOrientationV);
                lineHeight = childHeight;
                lineWidth = childWidth + spacingOrientationH;
            }
        }
        totalHeight += lineHeight + getPaddingBottom();
        if (MeasureSpec.EXACTLY != MeasureSpec.getMode(heightMeasureSpec)) {
            heightMeasureSpec = MeasureSpec.makeMeasureSpec(totalHeight, MeasureSpec.EXACTLY);
            setMeasuredDimension(widthMeasureSpec, heightMeasureSpec);
        }
    }

    @Override
    protected void onLayout(boolean changed, int l, int t, int r, int b) {
        int spacingOrientationV = getSpacingOrientationV();
        int spacingOrientationH = getSpacingOrientationH();
        int width = getMeasuredWidth();
        int totalHeight = getPaddingTop(), lineWidth = getPaddingLeft(), lineHeight = 0;
        for (int i = 0, size = getChildCount(); i < size; i ++) {
            View child = getChildAt(i);
            if (null == child || View.GONE == child.getVisibility()) {
                continue;
            }
            int childH = child.getMeasuredHeight();
            int childW = child.getMeasuredWidth();
            if (lineWidth + childW <= width) {
                child.layout(lineWidth, totalHeight, lineWidth + childW, totalHeight + childH);
                lineHeight = Math.max(lineHeight, childH);
                lineWidth += childW + spacingOrientationH;
            } else {
                totalHeight += lineHeight + (0 == i ? 0 : spacingOrientationV);
                child.layout(getPaddingLeft(), totalHeight, getPaddingLeft() + childW, totalHeight + childH);
                lineWidth = getPaddingLeft() + childW + spacingOrientationH;
                lineHeight = childH;
            }
        }
    }

    /**
     * */
    public void setSpacingOrientationV(int spacing) {
        this.mSpacingOrientationV = spacing;
        requestLayout();
    }

    /**
     * */
    public int getSpacingOrientationV() {
        return this.mSpacingOrientationV;
    }

    /**
     * */
    public void setSpacingOrientationH(int spacing) {
        this.mSpacingOrientationH = spacing;
        requestLayout();
    }

    /**
     * */
    public int getSpacingOrientationH() {
        return this.mSpacingOrientationH;
    }

    /**
     * */
    private int dp2value(float dp) {
        return (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP
                , dp, getResources().getDisplayMetrics());
    }

    /**
     * */
    public void setAdapter(Adapter adapter) {
        removeAllViews();
        if (null == adapter) {
            postInvalidate();
            return;
        }
        setAdapterInner(adapter);
    }

    /**
     * */
    public Adapter getAdapter() {
        return this.mAdapter;
    }

    /**
     * */
    private void setAdapterInner(Adapter adapter) {
        if (null != mAdapter) {
            mAdapter.unregisterAdapterDataObserver(mObserver);
        }
        if (null != adapter) {
            this.mAdapter = adapter;
            adapter.registerAdapterDataObserver(mObserver);
        }
        layoutAllChildren();
    }

    /**
     * */
    private void layoutAllChildren() {
        final Adapter adapter = getAdapter();
        if (null == adapter) {return;}
        removeAllViews();

        for (int i = 0, size = adapter.getCount(); i < size; i ++) {
            final View view = adapter.getView(i);
            if (null == view || View.GONE == view.getVisibility()) {
                continue;
            }
            LayoutParams layoutParams = new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
            addView(view, layoutParams);
        }
    }

    /**
     * */
    public static abstract class Adapter {

        /**
         * */
        private final AdapterDataObservable mObservable = new AdapterDataObservable();

        /**
         * */
        public void unregisterAdapterDataObserver(AdapterDataObserver observer) {
            mObservable.unregisterObserver(observer);
        }

        /**
         * */
        public void registerAdapterDataObserver(AdapterDataObserver observer) {
            mObservable.registerObserver(observer);
        }

        /***
         * */
        public final void notifyDataSetChanged() {
            mObservable.notifyChanged();
        }

        /***
         * */
        public abstract View getView(int position);

        /***
         * */
        public abstract int getCount();
    }

    /**
     * */
    private static class AdapterDataObservable extends Observable<AdapterDataObserver> {

        public void notifyChanged() {
            for (AdapterDataObserver observer : mObservers) {
                if (null == observer) {continue;}
                observer.onChanged();
            }
        }
    }

    /**
     * */
    private abstract class AdapterDataObserver {

        public abstract void onChanged();
    }

    /**
     * */
    private class TagsLayoutDataObserver extends AdapterDataObserver {

        @Override
        public void onChanged() {
            layoutAllChildren();
        }
    }
}
