/*
 * MIT License
 *
 * Copyright (c) 2017 Yuriy Budiyev [yuriy.budiyev@yandex.ru]
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.budiyev.android.imageloader;

import android.graphics.Bitmap;
import android.graphics.drawable.BitmapDrawable;
import android.support.annotation.MainThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import android.view.View;

/**
 * Image loading and displaying process callbacks
 */
public final class Callbacks<T> {
    private volatile LoadCallback<T> mLoadCallback;
    private volatile ErrorCallback<T> mErrorCallback;
    private volatile DisplayCallback<T> mDisplayCallback;

    public Callbacks() {
    }

    /**
     * Set image load callback
     *
     * @see LoadCallback#onLoaded(ImageSource, Bitmap)
     */
    @NonNull
    public Callbacks<T> load(@Nullable LoadCallback<T> callback) {
        mLoadCallback = callback;
        return this;
    }

    /**
     * Set image load error callback
     *
     * @see ErrorCallback#onError(ImageSource, Throwable)
     */
    @NonNull
    public Callbacks<T> error(@Nullable ErrorCallback<T> callback) {
        mErrorCallback = callback;
        return this;
    }

    /**
     * Set image display callback
     *
     * @see DisplayCallback#onDisplayed(ImageSource, BitmapDrawable, View)
     */
    @NonNull
    public Callbacks<T> display(@Nullable DisplayCallback<T> callback) {
        mDisplayCallback = callback;
        return this;
    }

    @WorkerThread
    static <T> void notifyLoaded(@Nullable Callbacks<T> callbacks, @NonNull ImageSource<T> source,
            @NonNull Bitmap image) {
        if (callbacks == null) {
            return;
        }
        LoadCallback<T> callback = callbacks.mLoadCallback;
        if (callback == null) {
            return;
        }
        callback.onLoaded(source, image);
    }

    @WorkerThread
    static <T> void notifyError(@Nullable Callbacks<T> callbacks, @NonNull ImageSource<T> source,
            @NonNull Throwable error) {
        if (callbacks == null) {
            return;
        }
        ErrorCallback<T> callback = callbacks.mErrorCallback;
        if (callback == null) {
            return;
        }
        callback.onError(source, error);
    }

    @MainThread
    static <T> void notifyDisplayed(@Nullable Callbacks<T> callbacks,
            @NonNull ImageSource<T> source, @NonNull BitmapDrawable drawable, @NonNull View view) {
        if (callbacks == null) {
            return;
        }
        DisplayCallback<T> callback = callbacks.mDisplayCallback;
        if (callback == null) {
            return;
        }
        callback.onDisplayed(source, drawable, view);
    }
}
